<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    /**
     * Display the admin dashboard.
     */
    /**
     * Display the admin dashboard.
     */
    public function index()
    {
        // 1. KPI Cards Data
        $totalUsers = \App\Models\User::count();
        $activeAds = \App\Models\Ad::where('status', 'Active')->count();
        
        // Revenue: Sum of all subscription amounts (this is a simple approximation)
        // Ideally we would sum up actual transaction logs
        $monthlyRevenue = \App\Models\Subscription::where('status', 'active')->sum('amount');
        
        $pendingAds = \App\Models\Ad::where('status', 'Pending')->count();

        $stats = [
            'total_users' => $totalUsers,
            'active_ads' => $activeAds,
            'monthly_revenue' => $monthlyRevenue,
            'pending_ads' => $pendingAds,
        ];

        // 2. Charts Data
        
        // Users by Plan - Dynamic based on user_plans table
        $usersByPlan = \App\Models\User::leftJoin('subscriptions', function($join) {
                $join->on('users.id', '=', 'subscriptions.user_id')
                     ->where('subscriptions.status', '=', 'active');
            })
            ->leftJoin('user_plans', 'subscriptions.plan_id', '=', 'user_plans.id')
            ->selectRaw('COALESCE(user_plans.plan_name, "Free") as plan_name, COUNT(DISTINCT users.id) as count')
            ->groupBy('user_plans.plan_name')
            ->orderBy('count', 'desc')
            ->pluck('count', 'plan_name');

        // Ad Categories Distribution - Top 10 Most Popular
        $adCategories = \App\Models\Ad::with('category')
            ->selectRaw('category_id, count(*) as count')
            ->whereNotNull('category_id')
            ->groupBy('category_id')
            ->orderByRaw('count DESC')
            ->limit(10)
            ->get()
            ->mapWithKeys(function ($item) {
                return [$item->category->name ?? 'Unknown' => $item->count];
            });

        // 3. Recent Activity Feed
        // Fetch latest 5 users
        $latestUsers = \App\Models\User::latest()->take(5)->get()->map(function($user) {
            return [
                'type' => 'user',
                'description' => 'New user registered',
                'subject' => $user->name,
                'status' => 'Active', // Users are active by default usually
                'created_at' => $user->created_at,
                'badge_class' => 'status-active'
            ];
        });

        // Fetch latest 5 ads
        $latestAds = \App\Models\Ad::with('user')->latest()->take(5)->get()->map(function($ad) {
            return [
                'type' => 'ad',
                'description' => 'New ad posted: ' . \Illuminate\Support\Str::limit($ad->title, 20),
                'subject' => $ad->user->name ?? 'Unknown User',
                'status' => $ad->status,
                'created_at' => $ad->created_at,
                'badge_class' => match($ad->status) {
                    'Active' => 'status-active',
                    'Pending' => 'status-pending',
                    'Rejected' => 'status-inactive',
                    default => 'status-pending'
                }
            ];
        });

        // Merge and sort
        $recentActivity = $latestUsers->concat($latestAds)
            ->sortByDesc('created_at')
            ->take(8); // Show top 8 mixed actions

        return view('admin.dashboard', compact('stats', 'usersByPlan', 'adCategories', 'recentActivity'));
    }

    /**
     * Get users by plan data for date range (AJAX endpoint)
     */
    public function getUsersByPlanData(Request $request)
    {
        $startDate = $request->input('start_date', now()->subDays(28)->format('Y-m-d'));
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        
        // Get all unique dates in the range
        $dates = [];
        $current = \Carbon\Carbon::parse($startDate);
        $end = \Carbon\Carbon::parse($endDate);
        
        while ($current <= $end) {
            $dates[] = $current->format('Y-m-d');
            $current->addDay();
        }
        
        // Get all plan names
        $plans = \App\Models\UserPlan::pluck('plan_name')->toArray();
        $plans[] = 'Free'; // Add Free for users without subscription
        
        // Initialize result structure
        $result = [];
        foreach ($plans as $plan) {
            $result[$plan] = array_fill_keys($dates, 0);
        }
        
        // Get cumulative user counts by plan for each date
        foreach ($dates as $date) {
            $usersByPlan = \App\Models\User::where('users.created_at', '<=', $date . ' 23:59:59')
                ->leftJoin('subscriptions', function($join) use ($date) {
                    $join->on('users.id', '=', 'subscriptions.user_id')
                         ->where('subscriptions.status', '=', 'active')
                         ->where('subscriptions.start_date', '<=', $date)
                         ->where(function($q) use ($date) {
                             $q->whereNull('subscriptions.end_date')
                               ->orWhere('subscriptions.end_date', '>=', $date);
                         });
                })
                ->leftJoin('user_plans', 'subscriptions.plan_id', '=', 'user_plans.id')
                ->selectRaw('COALESCE(user_plans.plan_name, "Free") as plan_name, COUNT(DISTINCT users.id) as count')
                ->groupBy('user_plans.plan_name')
                ->pluck('count', 'plan_name');
            
            foreach ($usersByPlan as $planName => $count) {
                if (isset($result[$planName])) {
                    $result[$planName][$date] = $count;
                }
            }
        }
        
        return response()->json([
            'dates' => $dates,
            'plans' => $result
        ]);
    }

    /**
     * Get user location data for world map (AJAX endpoint)
     */
    public function getUserLocationData(Request $request)
    {
        $planFilter = $request->input('plan', 'all');
        $startDate = $request->input('start_date', null);
        $endDate = $request->input('end_date', now()->format('Y-m-d'));
        
        $query = \App\Models\User::query();
        
        // Apply date filter (users who registered in this period)
        if ($startDate) {
            $query->whereBetween('created_at', [$startDate . ' 00:00:00', $endDate . ' 23:59:59']);
        }
        
        // Get users with location data
        $users = $query->whereNotNull('city')
            ->whereNotNull('country')
            ->get();
        
        // Group by city, country
        $locations = $users->groupBy(function($user) {
            return $user->city . '|' . $user->country;
        })->map(function($cityUsers, $key) use ($planFilter) {
            list($city, $country) = explode('|', $key);
            
            // Get first user's coordinates (assuming all users in same city have same coords)
            $firstUser = $cityUsers->first();
            $lat = $firstUser->latitude ?? $this->getCityDefaultCoords($city, $country)['lat'];
            $lng = $firstUser->longitude ?? $this->getCityDefaultCoords($city, $country)['lng'];
            
            // Count users by plan type
            $planCounts = [
                'free' => 0,
                'standard' => 0,
                'vip_buyer' => 0,
                'vip_seller' => 0,
                'vip_pro' => 0
            ];
            
            foreach ($cityUsers as $user) {
                // Get user's active subscription plan
                $subscription = $user->subscriptions()->where('status', 'active')->first();
                
                if (!$subscription) {
                    $planCounts['free']++;
                } else {
                    $planName = strtolower(str_replace(' ', '_', $subscription->plan->plan_name ?? ''));
                    if (isset($planCounts[$planName])) {
                        $planCounts[$planName]++;
                    } else {
                        $planCounts['standard']++; // Default fallback
                    }
                }
            }
            
            // Apply plan filter
            $totalCount = $cityUsers->count();
            if ($planFilter !== 'all') {
                $filterKey = $planFilter;
                if ($planFilter === 'premium') {
                    $totalCount = $planCounts['standard'] + $planCounts['vip_buyer'] + $planCounts['vip_seller'] + $planCounts['vip_pro'];
                } elseif ($planFilter === 'vip') {
                    $totalCount = $planCounts['vip_buyer'] + $planCounts['vip_seller'] + $planCounts['vip_pro'];
                } else {
                    $totalCount = $planCounts[$filterKey] ?? 0;
                }
                
                // Skip locations with zero users for this filter
                if ($totalCount == 0) {
                    return null;
                }
            }
            
            return [
                'city' => $city,
                'country' => $country,
                'lat' => (float) $lat,
                'lng' => (float) $lng,
                'total' => $totalCount,
                'plan_breakdown' => $planCounts
            ];
        })->filter()->values();
        
        return response()->json([
            'locations' => $locations,
            'total_users' => $locations->sum('total'),
            'period' => $startDate ? "$startDate to $endDate" : "Lifetime"
        ]);
    }
    
    /**
     * Get default coordinates for a city (fallback if lat/lng not in DB)
     */
    private function getCityDefaultCoords($city, $country)
    {
        // Major cities default coordinates (can be expanded)
        $cityCoords = [
            'Mumbai|India' => ['lat' => 19.0760, 'lng' => 72.8777],
            'Delhi|India' => ['lat' => 28.7041, 'lng' => 77.1025],
            'Bangalore|India' => ['lat' => 12.9716, 'lng' => 77.5946],
            'New York|USA' => ['lat' => 40.7128, 'lng' => -74.0060],
            'London|UK' => ['lat' => 51.5074, 'lng' => -0.1278],
            'Tokyo|Japan' => ['lat' => 35.6762, 'lng' => 139.6503],
            'Singapore|Singapore' => ['lat' => 1.3521, 'lng' => 103.8198],
            'Dubai|UAE' => ['lat' => 25.2048, 'lng' => 55.2708],
        ];
        
        $key = $city . '|' . $country;
        return $cityCoords[$key] ?? ['lat' => 0, 'lng' => 0]; // Default to 0,0 if not found
    }
}
