<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ChatMessage;
use App\Models\User;
use App\Models\GuardianLog;

class ChatController extends Controller
{
    public function index(Request $request)
    {
        $query = ChatMessage::query()->with(['sender', 'receiver', 'ad']);

        // Filter: Show only flagged/risky messages
        if ($request->has('flagged')) {
            $query->where('is_flagged', true);
        }

        // Search by content
        if ($request->filled('search')) {
            $query->where('message', 'like', '%' . $request->search . '%');
        }

        $messages = $query->latest()->paginate(20);

        return view('admin.chat.index', compact('messages'));
    }

    public function showConversation($user1, $user2)
    {
        $messages = ChatMessage::where(function($q) use ($user1, $user2) {
            $q->where('sender_id', $user1)->where('receiver_id', $user2);
        })->orWhere(function($q) use ($user1, $user2) {
            $q->where('sender_id', $user2)->where('receiver_id', $user1);
        })->orderBy('created_at', 'asc')->get();

        $u1 = User::findOrFail($user1);
        $u2 = User::findOrFail($user2);

        return view('admin.chat.conversation', compact('messages', 'u1', 'u2'));
    }

    public function flagMessage(Request $request, $id)
    {
        $message = ChatMessage::findOrFail($id);
        $message->update([
            'is_flagged' => true, 
            'flag_reason' => 'Manual Flag by Admin: ' . $request->reason
        ]);

        return back()->with('success', 'Message flagged successfully.');
    }

    public function banUser(Request $request, $id)
    {
        $user = User::findOrFail($id);
        $user->update(['status' => 'banned']);
        
        // Log it
        GuardianLog::create([
             'action' => 'MANUAL_BAN',
             'target_type' => 'User',
             'target_id' => $user->id,
             'reason' => 'Banned from Chat Monitor'
        ]);

        return back()->with('success', 'User has been banned.');
    }
}
