<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Banner;
use App\Models\BannerPackage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class BannerManagementController extends Controller
{
    public function index(Request $request)
    {
        $query = Banner::with(['user', 'package']);

        // Filter by status
        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        // Filter by created by (admin/user)
        if ($request->has('created_by')) {
            if ($request->created_by === 'admin') {
                $query->where('is_admin_created', true);
            } elseif ($request->created_by === 'user') {
                $query->where('is_admin_created', false);
            }
        }

        // Search
        if ($request->has('search')) {
            $query->where('title', 'like', '%' . $request->search . '%');
        }

        $banners = $query->orderBy('priority', 'desc')->paginate(20);

        // Calculate total revenue from transactions
        $totalRevenue = \App\Models\BannerTransaction::where('payment_status', 'completed')->sum('amount');

        return view('admin.banners.index', compact('banners', 'totalRevenue'));
    }

    public function create()
    {
        $packages = BannerPackage::where('is_active', true)->get();
        return view('admin.banners.create', compact('packages'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'image' => 'required|image|max:2048',
            'redirect_type' => 'required|in:category,package,plan,url,ad,user_profile,static',
            'redirect_value' => 'nullable|string',
            'target_location_type' => 'required|in:all,country,state,city',
            'target_countries' => 'nullable|array',
            'target_states' => 'nullable|array',
            'target_cities' => 'nullable|array',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',
            'priority' => 'nullable|integer|min:0',
            'status' => 'required|in:draft,active,paused',
        ]);

        // Upload image
        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('banners', 'public');
            $validated['image_url'] = Storage::url($path);
        }

        $validated['is_admin_created'] = true;
        $validated['billing_model'] = 'impression';
        $validated['total_budget'] = 0; // Admin banners don't have budget

        Banner::create($validated);

        return redirect()->route('admin.banners.index')
            ->with('success', 'Banner created successfully!');
    }

    public function edit(Banner $banner)
    {
        $packages = BannerPackage::where('is_active', true)->get();
        return view('admin.banners.edit', compact('banner', 'packages'));
    }

    public function update(Request $request, Banner $banner)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'image' => 'nullable|image|max:2048',
            'redirect_type' => 'required|in:category,package,plan,url,ad,user_profile,static',
            'redirect_value' => 'nullable|string',
            'target_location_type' => 'required|in:all,country,state,city',
            'target_countries' => 'nullable|array',
            'target_states' => 'nullable|array',
            'target_cities' => 'nullable|array',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after:start_date',
            'priority' => 'nullable|integer|min:0',
            'status' => 'required|in:draft,active,paused,completed',
        ]);

        // Upload new image if provided
        if ($request->hasFile('image')) {
            // Delete old image
            if ($banner->image_url) {
                Storage::disk('public')->delete(str_replace('/storage/', '', $banner->image_url));
            }
            
            $path = $request->file('image')->store('banners', 'public');
            $validated['image_url'] = Storage::url($path);
        }

        $banner->update($validated);

        return redirect()->route('admin.banners.index')
            ->with('success', 'Banner updated successfully!');
    }

    public function destroy(Banner $banner)
    {
        // Delete image
        if ($banner->image_url) {
            Storage::disk('public')->delete(str_replace('/storage/', '', $banner->image_url));
        }

        $banner->delete();

        return redirect()->route('admin.banners.index')
            ->with('success', 'Banner deleted successfully!');
    }

    public function approve(Banner $banner)
    {
        $banner->update([
            'status' => 'active',
            'admin_notes' => 'Approved by admin',
        ]);

        return redirect()->back()
            ->with('success', 'Banner approved!');
    }

    public function reject(Request $request, Banner $banner)
    {
        $banner->update([
            'status' => 'rejected',
            'admin_notes' => $request->input('reason', 'Rejected by admin'),
        ]);

        return redirect()->back()
            ->with('success', 'Banner rejected!');
    }

    public function pause(Banner $banner)
    {
        $banner->update(['status' => 'paused']);

        return redirect()->back()
            ->with('success', 'Banner paused!');
    }

    public function resume(Banner $banner)
    {
        $banner->update(['status' => 'active']);

        return redirect()->back()
            ->with('success', 'Banner resumed!');
    }

    public function analytics()
    {
        $stats = [
            'total_banners' => Banner::count(),
            'active_banners' => Banner::where('status', 'active')->count(),
            'total_impressions' => Banner::sum('total_impressions'),
            'total_clicks' => Banner::sum('total_clicks'),
            'average_ctr' => Banner::where('total_impressions', '>', 0)->avg('ctr'),
            'total_revenue' => Banner::where('is_admin_created', false)->sum('spent_amount'),
        ];

        $topBanners = Banner::orderBy('total_clicks', 'desc')->limit(10)->get();

        return view('admin.banners.analytics', compact('stats', 'topBanners'));
    }
    public function reorder(Request $request)
    {
        $request->validate([
            'items' => 'required|array',
            'items.*' => 'exists:banners,id',
        ]);

        foreach ($request->items as $index => $id) {
            // Assign priority based on order (higher index = lower priority, or reverse)
            // Let's assume top of list = highest priority.
            // If list has 20 items, index 0 gets priority 20, index 19 gets priority 1.
            $count = count($request->items);
            Banner::where('id', $id)->update(['priority' => $count - $index]);
        }

        return response()->json(['success' => true]);
    }

    public function toggleStatus(Banner $banner)
    {
        $newStatus = $banner->status === 'active' ? 'paused' : 'active';
        $banner->update(['status' => $newStatus]);

        return response()->json([
            'success' => true, 
            'status' => $newStatus,
            'message' => 'Banner status updated to ' . ucfirst($newStatus)
        ]);
    }
}
