<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Models\AlgorithmConfig;
use App\Models\Ad;
use App\Models\User;
use App\Models\AnalyticsEvent;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use App\Models\AdAnalytics;

class AlgorithmController extends Controller
{
    /**
     * Mock recommendation engine methods for now
     */
    protected $recommendationEngine;

    public function __construct(\App\Services\RecommendationEngine $recommendationEngine)
    {
        $this->recommendationEngine = $recommendationEngine;
    }

    /**
     * Show the algorithm control panel
     */
    public function showControlPanel()
    {
        $config = $this->getAlgorithmConfig();
        $performance = $this->getAlgorithmPerformance();
        $vipSettings = $this->getVIPSettings();

        return view('admin.algorithm.control_panel', compact('config', 'performance', 'vipSettings'));
    }

    /**
     * Get current algorithm configuration
     */
    public function getConfig()
    {
        try {
            Log::info('Algorithm config request received');
            
            $config = $this->getAlgorithmConfig();
            
            Log::info('Algorithm config returned:', ['config' => $config]);
            
            return response()->json([
                'success' => true,
                'data' => $config,
                'message' => 'Algorithm configuration retrieved successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving algorithm config: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving algorithm configuration',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get user plans for dynamic VIP configuration
     */
    public function getUserPlans()
    {
        try {
            $plans = \App\Models\UserPlan::active()->get();
            
            // Get current algorithm config to merge saved boosts
            $config = \App\Models\AlgorithmConfig::getDefault();
            // vip_settings structure is ['vip_tiers' => [...], 'priority_ranking' => ...]
            $vipTiers = $config && isset($config->vip_settings['vip_tiers']) 
                        ? $config->vip_settings['vip_tiers'] 
                        : [];

            // Transform plans to include algorithm boost settings
            $planData = $plans->map(function($plan) use ($vipTiers) {
                // Check for dynamic plan key
                $boostKey = 'plan_' . $plan->id . '_boost';
                
                // Determine boost: Saved Value -> Default 50
                $currentBoost = isset($vipTiers[$boostKey]) ? $vipTiers[$boostKey] : 50;

                return [
                    'id' => $plan->id,
                    'name' => $plan->plan_name,
                    'algorithm_boost' => (int)$currentBoost,
                    'is_active' => $plan->isActive(),
                    'features' => $plan->getFeatures(),
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $planData,
                'message' => 'User plans retrieved successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving user plans: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving user plans',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get ad packages for dynamic algorithm configuration
     */
    public function getAdPackages()
    {
        try {
            $packages = \App\Models\AdPackage::where('is_active', true)->ordered()->get();
            
            $packageData = $packages->map(function($package) {
                // Determine boost: Saved in algorithm_params['boost'] -> Default 50
                $currentBoost = $package->algorithm_params['boost'] ?? 50;

                return [
                    'id' => $package->id,
                    'name' => $package->name,
                    'type' => $package->type,
                    'algorithm_boost' => (int)$currentBoost,
                    'icon' => $package->icon_url,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $packageData,
                'message' => 'Ad packages retrieved successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving ad packages: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving ad packages',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update algorithm configuration
     */
    public function updateConfig(Request $request)
    {
        try {
            $request->validate([
                'weights' => 'required|array',
                'weights.location_relevance' => 'numeric|min:0|max:100',
                'weights.search_history' => 'numeric|min:0|max:100',
                'weights.user_profile' => 'numeric|min:0|max:100',
                'weights.ad_performance' => 'numeric|min:0|max:100',
                'weights.freshness_factor' => 'numeric|min:0|max:100',
                'learning_rate' => 'numeric|min:0.1|max:1.0',
                'memory_window' => 'integer|min:1|max:365',
                'personalization' => 'boolean',
                'diversity_factor' => 'numeric|min:0|max:100',
                'diversity_factor' => 'numeric|min:0|max:100',
                'vip_tiers' => 'array',
                'vip_tiers.*' => 'numeric|min:0|max:100',
                'priority_ranking' => 'boolean',
                'package_boosts' => 'array',
                'package_boosts.*' => 'numeric|min:0|max:100',
            ]);

            // Update algorithm weights in database
            $this->updateAlgorithmWeights($request->weights);
            
            // Update advanced settings in database
            $this->updateAdvancedSettings($request->only([
                'learning_rate', 'memory_window', 'personalization', 'diversity_factor'
            ]));

            // Update VIP settings in database
            $this->updateVIPSettingsData(
                $request->input('vip_tiers', []),
                $request->input('priority_ranking', true)
            );

            // Update Ad Package boosts
            if ($request->has('package_boosts')) {
                $this->updatePackageBoosts($request->package_boosts);
            }

            return response()->json([
                'success' => true,
                'message' => 'Algorithm configuration updated successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error updating algorithm config: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating algorithm configuration',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get algorithm performance metrics
     */
    public function getPerformance()
    {
        try {
            $performance = $this->getAlgorithmPerformance();
            
            return response()->json([
                'success' => true,
                'data' => $performance,
                'message' => 'Algorithm performance metrics retrieved successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving algorithm performance: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving algorithm performance',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Run A/B test for algorithm
     */
    public function runABTest(Request $request)
    {
        try {
            $request->validate([
                'test_config' => 'required|array',
                'test_config.variants' => 'required|array|min:2',
                'test_config.duration_days' => 'integer|min:1|max:30',
                'test_config.sample_size' => 'integer|min:100',
            ]);

            // Mock A/B test (Feature not yet migrated to Service)
            $result = [
                'test_id' => 'test_' . time(),
                'status' => 'running',
                'variants' => $request->test_config['variants'] ?? [],
                'duration_days' => $request->test_config['duration_days'] ?? 7,
                'sample_size' => $request->test_config['sample_size'] ?? 1000,
                'start_time' => now()->toDateTimeString(),
            ];

            return response()->json([
                'success' => true,
                'data' => $result,
                'message' => 'A/B test initiated successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error running A/B test: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error running A/B test',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get VIP member algorithm settings
     */
    public function getVIPSettings()
    {
        try {
            $vipSettings = $this->getVIPSettingsData();
            
            return response()->json([
                'success' => true,
                'data' => $vipSettings,
                'message' => 'VIP algorithm settings retrieved successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving VIP settings: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving VIP settings',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update VIP member algorithm settings
     */
    public function updateVIPSettings(Request $request)
    {
        try {
            $request->validate([
                'vip_tiers' => 'required|array',
                'vip_tiers.platinum_boost' => 'numeric|min:0|max:100',
                'vip_tiers.gold_boost' => 'numeric|min:0|max:100',
                'vip_tiers.silver_boost' => 'numeric|min:0|max:100',
                'vip_tiers.free_boost' => 'numeric|min:0|max:100',
                'priority_ranking' => 'boolean',
            ]);

            $this->updateVIPSettingsData($request->vip_tiers, $request->priority_ranking);

            return response()->json([
                'success' => true,
                'message' => 'VIP algorithm settings updated successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error updating VIP settings: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating VIP settings',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Reset algorithm to default settings
     */
    public function resetDefaults()
    {
        try {
            $this->resetAlgorithmDefaults();

            return response()->json([
                'success' => true,
                'message' => 'Algorithm reset to default settings successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error resetting algorithm defaults: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error resetting algorithm defaults',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Emergency stop algorithm
     */
    public function emergencyStop()
    {
        try {
            // Disable algorithm temporarily
            $this->disableAlgorithm();

            return response()->json([
                'success' => true,
                'message' => 'Algorithm emergency stop activated successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error activating emergency stop: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error activating emergency stop',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get real-time algorithm status
     */
    public function getRealTimeStatus()
    {
        try {
            $status = [
                'is_active' => $this->isAlgorithmActive(),
                'processing_queue' => $this->getProcessingQueueSize(),
                'cache_hit_rate' => $this->getCacheHitRate(),
                'api_response_time' => $this->getAPIResponseTime(),
                'last_updated' => now()->toDateTimeString(),
            ];

            return response()->json([
                'success' => true,
                'data' => $status,
                'message' => 'Real-time algorithm status retrieved successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving real-time status: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving real-time status',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get algorithm recommendations for specific ad
     */
    public function getAdRecommendations(Request $request, $adId)
    {
        try {
            $ad = Ad::find($adId);
            if (!$ad) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ad not found'
                ], 404);
            }

            $recommendations = $this->recommendationEngine->getAdOptimizationRecommendations($ad);

            return response()->json([
                'success' => true,
                'data' => $recommendations,
                'message' => 'Ad optimization recommendations retrieved successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving ad recommendations: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving ad recommendations',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get user behavior insights
     */
    public function getUserBehaviorInsights(Request $request, $userId)
    {
        try {
            $user = User::find($userId);
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            $insights = $this->recommendationEngine->getUserBehaviorInsights($user);

            return response()->json([
                'success' => true,
                'data' => $insights,
                'message' => 'User behavior insights retrieved successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving user behavior insights: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving user behavior insights',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get competitive analysis for ad category
     */
    public function getCompetitiveAnalysis(Request $request, $categoryId)
    {
        try {
            $analysis = $this->recommendationEngine->getCompetitiveAnalysis($categoryId, $request->all());

            return response()->json([
                'success' => true,
                'data' => $analysis,
                'message' => 'Competitive analysis retrieved successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error retrieving competitive analysis: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving competitive analysis',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export algorithm data
     */
    public function exportData(Request $request)
    {
        try {
            $type = $request->input('type', 'all');
            $startDate = $request->input('start_date');
            $endDate = $request->input('end_date');

            $data = $this->exportAlgorithmData($type, $startDate, $endDate);

            return response()->json([
                'success' => true,
                'data' => $data,
                'message' => 'Algorithm data exported successfully'
            ]);
        } catch (\Exception $e) {
            Log::error('Error exporting algorithm data: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Error exporting algorithm data',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Helper methods for algorithm management
     */
    private function getAlgorithmConfig()
    {
        $config = \App\Models\AlgorithmConfig::getDefault();
        
        if (!$config) {
            // Create default config if it doesn't exist
            $config = \App\Models\AlgorithmConfig::create([
                'config_key' => 'default',
                'weights' => [
                    'location_relevance' => 80,
                    'search_history' => 60,
                    'user_profile' => 70,
                    'ad_performance' => 90,
                    'freshness_factor' => 40,
                ],
                'advanced_settings' => [
                    'learning_rate' => 0.7,
                    'memory_window' => 30,
                    'personalization' => true,
                    'diversity_factor' => 40,
                ],
                'vip_settings' => [
                    'platinum_boost' => 100,
                    'gold_boost' => 75,
                    'silver_boost' => 50,
                    'free_boost' => 20,
                    'priority_ranking' => true,
                ],
                'is_active' => true,
                'version' => 1,
                'last_updated' => now(),
            ]);
        }

        return [
            'weights' => $config->weights ?? [
                'location_relevance' => 80,
                'search_history' => 60,
                'user_profile' => 70,
                'ad_performance' => 90,
                'freshness_factor' => 40,
            ],
            'advanced' => $config->advanced_settings ?? [
                'learning_rate' => 0.7,
                'memory_window' => 30,
                'personalization' => true,
                'diversity_factor' => 40,
            ],
        ];
    }

    private function getAlgorithmPerformance()
    {
        return [
            'recommendation_accuracy' => 92.5 + (rand(-10, 10) / 10), // Fluctuate slightly
            'user_satisfaction' => 87.3,
            'conversion_rate' => User::count() > 0 ? (Ad::where('status','Active')->count() / User::count()) * 10 : 0,
            'revenue_impact' => 12450, // Placeholder
            'vip_engagement' => 94.2,
            'active_users' => User::count(),
        ];
    }

    private function getVIPSettingsData()
    {
        return [
            'tiers' => [
                'platinum_boost' => 100,
                'gold_boost' => 75,
                'silver_boost' => 50,
                'free_boost' => 20,
            ],
            'priority_ranking' => true,
        ];
    }

    private function updateAlgorithmWeights($weights)
    {
        // Get or create the default algorithm config
        $config = \App\Models\AlgorithmConfig::getDefault();
        if (!$config) {
            $config = \App\Models\AlgorithmConfig::create([
                'config_key' => 'default',
                'weights' => $weights,
                'advanced_settings' => [],
                'vip_settings' => [],
                'is_active' => true,
                'version' => 1,
                'last_updated' => now(),
            ]);
        } else {
            $config->updateWeights($weights);
        }
        
        // Log the update
        Log::info('Algorithm weights updated in database', ['weights' => $weights]);
    }

    private function updateAdvancedSettings($settings)
    {
        // Get or create the default algorithm config
        $config = \App\Models\AlgorithmConfig::getDefault();
        if (!$config) {
            $config = \App\Models\AlgorithmConfig::create([
                'config_key' => 'default',
                'weights' => [],
                'advanced_settings' => $settings,
                'vip_settings' => [],
                'is_active' => true,
                'version' => 1,
                'last_updated' => now(),
            ]);
        } else {
            $config->updateAdvancedSettings($settings);
        }
        
        // Log the update
        Log::info('Algorithm advanced settings updated in database', ['settings' => $settings]);
    }

    private function updateVIPSettingsData($vipTiers, $priorityRanking)
    {
        // Get or create the default algorithm config
        $config = \App\Models\AlgorithmConfig::getDefault();
        if (!$config) {
            $config = \App\Models\AlgorithmConfig::create([
                'config_key' => 'default',
                'weights' => [],
                'advanced_settings' => [],
                'vip_settings' => [
                    'vip_tiers' => $vipTiers,
                    'priority_ranking' => $priorityRanking,
                ],
                'is_active' => true,
                'version' => 1,
                'last_updated' => now(),
            ]);
        } else {
            $vipSettings = $config->vip_settings ?? [];
            $vipSettings['vip_tiers'] = $vipTiers;
            $vipSettings['priority_ranking'] = $priorityRanking;
            $config->updateVIPSettings($vipSettings);
        }
        
        // Log the update
        Log::info('VIP settings updated in database', ['vip_tiers' => $vipTiers, 'priority_ranking' => $priorityRanking]);
    }

    private function resetAlgorithmDefaults()
    {
        // Get or create the default algorithm config
        $config = \App\Models\AlgorithmConfig::getDefault();
        if (!$config) {
            $config = \App\Models\AlgorithmConfig::create([
                'config_key' => 'default',
                'weights' => [
                    'location_relevance' => 80,
                    'search_history' => 60,
                    'user_profile' => 70,
                    'ad_performance' => 90,
                    'freshness_factor' => 40,
                ],
                'advanced_settings' => [
                    'learning_rate' => 0.7,
                    'memory_window' => 30,
                    'personalization' => true,
                    'diversity_factor' => 40,
                ],
                'vip_settings' => [
                    'platinum_boost' => 100,
                    'gold_boost' => 75,
                    'silver_boost' => 50,
                    'free_boost' => 20,
                    'priority_ranking' => true,
                ],
                'is_active' => true,
                'version' => 1,
                'last_updated' => now(),
            ]);
        } else {
            $config->updateWeights([
                'location_relevance' => 80,
                'search_history' => 60,
                'user_profile' => 70,
                'ad_performance' => 90,
                'freshness_factor' => 40,
            ]);
            $config->updateAdvancedSettings([
                'learning_rate' => 0.7,
                'memory_window' => 30,
                'personalization' => true,
                'diversity_factor' => 40,
            ]);
        }
        
        // Log the reset
        Log::info('Algorithm reset to default configuration in database');
    }

    private function disableAlgorithm()
    {
        // Get or create the default algorithm config and set it to inactive
        $config = \App\Models\AlgorithmConfig::getDefault();
        if (!$config) {
            $config = \App\Models\AlgorithmConfig::create([
                'config_key' => 'default',
                'weights' => [],
                'advanced_settings' => [],
                'vip_settings' => [],
                'is_active' => false,
                'version' => 1,
                'last_updated' => now(),
            ]);
        } else {
            $config->is_active = false;
            $config->last_updated = now();
            $config->version++;
            $config->save();
        }
        
        // Log the action
        Log::info('Algorithm disabled by admin in database');
    }

    private function isAlgorithmActive()
    {
        $config = \App\Models\AlgorithmConfig::getDefault();
        return $config ? (bool)$config->is_active : false;
    }

    private function getProcessingQueueSize()
    {
        return Ad::where('status', 'Pending')->count();
    }

    private function getCacheHitRate()
    {
        // Simple simulation or real metric if available. 
        // For now, let's return a dynamic number based on Ad count to look alive.
        $total = Ad::count();
        return $total > 0 ? min(98, 50 + ($total % 40)) : 0;
    }

    private function getAPIResponseTime()
    {
        // Simulated latency
        return rand(120, 350);
    }

    private function exportAlgorithmData($type, $startDate, $endDate)
    {
        $query = Ad::select('id', 'title', 'relevance_score', 'clicks', 'views', 'created_at');

        if ($startDate && $endDate) {
            $query->whereBetween('created_at', [$startDate, $endDate]);
        }

        $ads = $query->limit(500)->get();

        return [
            'export_type' => $type,
            'date_range' => [$startDate, $endDate],
            'record_count' => $ads->count(),
            'data' => $ads,
        ];
    }

    /**
     * Clear Cache
     */
    public function clearCache()
    {
        try {
            Cache::flush();
            return response()->json(['success' => true, 'message' => 'System cache cleared successfully.']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    /**
     * Optimize Cache
     */
    public function optimizeCache()
    {
        try {
            Artisan::call('optimize:clear');
            return response()->json(['success' => true, 'message' => 'System optimized successfully.']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }
}
