<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Ad;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AdStatisticsController extends Controller
{
    /**
     * Get main statistics for the ads dashboard.
     */
    public function getStats(Request $request)
    {
        // Base query with optional filtering
        $query = Ad::query();
        
        // Apply filters if provided
        if ($request->filled('country')) {
            $query->where('location_country', $request->country);
        }
        
        if ($request->filled('currency')) {
            $query->where('currency', $request->currency);
        }
        
        if ($request->filled('category')) {
            $query->where('category_id', $request->category);
        }
        
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('type')) {
            if ($request->type === 'featured') {
                $query->where('is_featured', true);
            } elseif ($request->type === 'regular') {
                $query->where('is_featured', false);
            }
        }

        // Get total count
        $total = $query->count();
        
        // Get status breakdown
        $statusStats = $query->select('status', DB::raw('count(*) as count'))
                            ->groupBy('status')
                            ->pluck('count', 'status');
        
        // Get featured count
        $featured = $query->where('is_featured', true)->count();
        
        // Get location counts with null checks
        $countries = $query->select('location_country')
                          ->distinct()
                          ->whereNotNull('location_country')
                          ->count();
        
        $states = $query->select('location_state')
                       ->distinct()
                       ->whereNotNull('location_state')
                       ->whereNotNull('location_country')
                       ->count();
        
        $cities = $query->select('location_city')
                       ->distinct()
                       ->whereNotNull('location_city')
                       ->whereNotNull('location_state')
                       ->whereNotNull('location_country')
                       ->count();

        // Get category breakdown
        $categoryStats = $query->select('category_id', DB::raw('count(*) as count'))
                              ->whereNotNull('category_id')
                              ->groupBy('category_id')
                              ->pluck('count', 'category_id');

        // Get currency breakdown
        $currencyStats = $query->select('currency', DB::raw('count(*) as count'))
                              ->whereNotNull('currency')
                              ->groupBy('currency')
                              ->pluck('count', 'currency');

        return response()->json([
            'total' => $total,
            'active' => $statusStats->get('Active', 0),
            'pending' => $statusStats->get('Pending', 0),
            'sold' => $statusStats->get('Sold', 0),
            'expired' => $statusStats->get('Expired', 0),
            'rejected' => $statusStats->get('Rejected', 0),
            'featured' => $featured,
            'countries' => $countries,
            'states' => $states,
            'cities' => $cities,
            'category_breakdown' => $categoryStats,
            'currency_breakdown' => $currencyStats
        ]);
    }

    /**
     * Get chart data for the ads dashboard.
     */
    public function getCharts(Request $request)
    {
        // Base query with optional filtering
        $query = Ad::query();
        
        // Apply filters if provided
        if ($request->filled('country')) {
            $query->where('location_country', $request->country);
        }
        
        if ($request->filled('currency')) {
            $query->where('currency', $request->currency);
        }
        
        if ($request->filled('category')) {
            $query->where('category_id', $request->category);
        }
        
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('type')) {
            if ($request->type === 'featured') {
                $query->where('is_featured', true);
            } elseif ($request->type === 'regular') {
                $query->where('is_featured', false);
            }
        }

        // Status distribution
        $statusData = $query->select('status', DB::raw('count(*) as count'))
                           ->groupBy('status')
                           ->orderBy('count', 'desc')
                           ->get();

        // Category distribution with category names
        $categoryData = $query->select('category_id', DB::raw('count(*) as count'))
                             ->whereNotNull('category_id')
                             ->groupBy('category_id')
                             ->orderBy('count', 'desc')
                             ->get();

        // Featured vs Regular
        $featuredData = $query->select('is_featured', DB::raw('count(*) as count'))
                             ->groupBy('is_featured')
                             ->get();

        // Location distribution based on level
        $locationLevel = $request->get('location_level', 'country');
        $locationData = $this->getLocationData($query, $locationLevel);

        // Top locations table
        $topLocations = $this->getTopLocations($query, $locationLevel);

        return response()->json([
            'status_distribution' => $statusData,
            'category_distribution' => $categoryData,
            'featured_distribution' => $featuredData,
            'location_distribution' => $locationData,
            'top_locations' => $topLocations
        ]);
    }

    /**
     * Get location data based on the specified level.
     */
    protected function getLocationData($query, $level)
    {
        switch ($level) {
            case 'country':
                return $query->select('location_country as name', DB::raw('count(*) as count'))
                            ->whereNotNull('location_country')
                            ->groupBy('location_country')
                            ->orderBy('count', 'desc')
                            ->limit(10)
                            ->get();
            
            case 'state':
                return $query->select('location_state as name', 'location_country as country', DB::raw('count(*) as count'))
                            ->whereNotNull('location_state')
                            ->whereNotNull('location_country')
                            ->groupBy('location_state', 'location_country')
                            ->orderBy('count', 'desc')
                            ->limit(10)
                            ->get();
            
            case 'city':
                return $query->select('location_city as name', 'location_state as state', 'location_country as country', DB::raw('count(*) as count'))
                            ->whereNotNull('location_city')
                            ->whereNotNull('location_state')
                            ->whereNotNull('location_country')
                            ->groupBy('location_city', 'location_state', 'location_country')
                            ->orderBy('count', 'desc')
                            ->limit(10)
                            ->get();
            
            default:
                return $query->select('location_country as name', DB::raw('count(*) as count'))
                            ->whereNotNull('location_country')
                            ->groupBy('location_country')
                            ->orderBy('count', 'desc')
                            ->limit(10)
                            ->get();
        }
    }

    /**
     * Get top locations for the table.
     */
    protected function getTopLocations($query, $level)
    {
        switch ($level) {
            case 'country':
                return $query->select(
                    'location_country as name',
                    DB::raw("'Country' as type"),
                    DB::raw('count(*) as total'),
                    DB::raw('SUM(CASE WHEN status = "Active" THEN 1 ELSE 0 END) as active'),
                    DB::raw('SUM(CASE WHEN is_featured = 1 THEN 1 ELSE 0 END) as featured'),
                    DB::raw('COALESCE(SUM(views), 0) as views')
                )
                ->whereNotNull('location_country')
                ->groupBy('location_country')
                ->orderBy('total', 'desc')
                ->limit(10)
                ->get();
            
            case 'state':
                return $query->select(
                    'location_state as name',
                    DB::raw("'State/Province' as type"),
                    'location_country as country',
                    DB::raw('count(*) as total'),
                    DB::raw('SUM(CASE WHEN status = "Active" THEN 1 ELSE 0 END) as active'),
                    DB::raw('SUM(CASE WHEN is_featured = 1 THEN 1 ELSE 0 END) as featured'),
                    DB::raw('COALESCE(SUM(views), 0) as views')
                )
                ->whereNotNull('location_state')
                ->whereNotNull('location_country')
                ->groupBy('location_state', 'location_country')
                ->orderBy('total', 'desc')
                ->limit(10)
                ->get();
            
            case 'city':
                return $query->select(
                    'location_city as name',
                    DB::raw("'City' as type"),
                    'location_state as state',
                    'location_country as country',
                    DB::raw('count(*) as total'),
                    DB::raw('SUM(CASE WHEN status = "Active" THEN 1 ELSE 0 END) as active'),
                    DB::raw('SUM(CASE WHEN is_featured = 1 THEN 1 ELSE 0 END) as featured'),
                    DB::raw('COALESCE(SUM(views), 0) as views')
                )
                ->whereNotNull('location_city')
                ->whereNotNull('location_state')
                ->whereNotNull('location_country')
                ->groupBy('location_city', 'location_state', 'location_country')
                ->orderBy('total', 'desc')
                ->limit(10)
                ->get();
            
            default:
                return $query->select(
                    'location_country as name',
                    DB::raw("'Country' as type"),
                    DB::raw('count(*) as total'),
                    DB::raw('SUM(CASE WHEN status = "Active" THEN 1 ELSE 0 END) as active'),
                    DB::raw('SUM(CASE WHEN is_featured = 1 THEN 1 ELSE 0 END) as featured'),
                    DB::raw('COALESCE(SUM(views), 0) as views')
                )
                ->whereNotNull('location_country')
                ->groupBy('location_country')
                ->orderBy('total', 'desc')
                ->limit(10)
                ->get();
        }
    }

    /**
     * Get real-time statistics that update after admin actions.
     */
    public function getRealTimeStats(Request $request)
    {
        return $this->getStats($request);
    }
}
