<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Ad;
use App\Models\Notification;
use App\Models\Setting;
use App\Models\AdAttributeValue;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class AdController extends Controller
{
    /**
     * Display a listing of ads with enhanced filtering.
     */
    public function index(Request $request)
    {
        $query = Ad::with(['user', 'category', 'subcategory', 'images'])
                    ->orderBy('created_at', 'desc');

        // Enhanced filtering for global management
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('country')) {
            $query->where('location_country', $request->country);
        }

        if ($request->filled('currency')) {
            $query->where('currency', $request->currency);
        }

        if ($request->filled('category')) {
            $query->where('category_id', $request->category);
        }

        if ($request->filled('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhereHas('user', function($userQuery) use ($search) {
                      $userQuery->where('name', 'like', "%{$search}%")
                               ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        $ads = $query->paginate($request->per_page ?? 25);

        // Get comprehensive stats for dashboard
        $stats = [
            'total' => Ad::count(),
            'active' => Ad::where('status', 'Active')->count(),
            'pending' => Ad::where('status', 'Pending')->count(),
            'expired' => Ad::where('status', 'Expired')->count(),
            'sold' => Ad::where('status', 'Sold')->count(),
            'featured' => Ad::where('is_featured', true)->count(),
        ];

        // Get distribution data
        $statusDistribution = Ad::select('status', \DB::raw('count(*) as count'))
            ->groupBy('status')
            ->get()
            ->pluck('count', 'status');

        $categoryDistribution = Ad::select('category_id', \DB::raw('count(*) as count'))
            ->whereNotNull('category_id')
            ->groupBy('category_id')
            ->with('category:id,name')
            ->get()
            ->mapWithKeys(function ($item) {
                return [optional($item->category)->name ?? 'Unknown' => $item->count];
            });

        $locationDistribution = Ad::select('location_city', \DB::raw('count(*) as count'))
            ->whereNotNull('location_city')
            ->groupBy('location_city')
            ->orderBy('count', 'desc')
            ->limit(10)
            ->get()
            ->pluck('count', 'location_city');

        // Get location counts
        $locationCounts = [
            'countries' => Ad::select('location_country')->distinct()->whereNotNull('location_country')->count(),
            'states' => Ad::select('location_state')->distinct()->whereNotNull('location_state')->count(),
            'cities' => Ad::select('location_city')->distinct()->whereNotNull('location_city')->count(),
        ];

       // Get countries and currencies for filtering
        $countries = Ad::select('location_country')->distinct()->pluck('location_country')->filter();
        $currencies = Ad::select('currency')->distinct()->pluck('currency')->filter();

        return view('admin.ads.index', compact(
            'ads', 
            'stats', 
            'countries', 
            'currencies',
            'statusDistribution',
            'categoryDistribution',
            'locationDistribution',
            'locationCounts'
        ));
    }

    /**
     * Show the form for creating a new ad.
     */
    public function create()
    {
        $categories = \App\Models\Category::all();
        $users = \App\Models\User::all();
        return view('admin.ads.create', compact('categories', 'users'));
    }

    /**
     * Store a newly created ad in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'price' => 'required|numeric|min:0',
            'category_id' => 'required|exists:categories,id',
            'user_id' => 'required|exists:users,id',
            'status' => 'required|in:Active,Pending,Draft',
            'location_city' => 'required|string',
            'location_state' => 'required|string',
            'location_country' => 'required|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        // Map UI fields to DB fields if needed
        $data = $request->all();
        $data['currency'] = $data['currency'] ?? 'USD';
        $data['is_active'] = ($data['status'] === 'Active');
        
        // Ensure city/state are synced for API compatibility (the issue we found earlier)
        $data['city'] = $data['location_city'];
        $data['state'] = $data['location_state'];
        $data['country'] = $data['location_country'];

        $ad = Ad::create($data);

        return redirect()->route('admin.ads.index')->with('success', 'Ad created successfully!');
    }

    /**
     * Approve an ad.
     */
    public function approve($id)
    {
        $ad = Ad::findOrFail($id);
        
        if ($ad->status === 'Active') {
            return response()->json(['success' => false, 'message' => 'Ad is already active!']);
        }

        $ad->update([
            'status' => 'Active',
            'approved_at' => now(),
            'approved_by' => auth()->id(),
        ]);

        // Send approval notification to user
        Notification::create([
            'user_id' => $ad->user_id,
            'type' => 'ad_approved',
            'title' => 'Your Ad Has Been Approved',
            'message' => "Your ad '{$ad->title}' has been approved and is now live!",
            'data' => [
                'ad_id' => $ad->id,
                'ad_title' => $ad->title,
                'ad_url' => route('ads.show', $ad->id)
            ]
        ]);

        return response()->json(['success' => true, 'message' => 'Ad approved successfully!']);
    }

    /**
     * Reject an ad.
     */
    public function reject(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'rejection_reason' => 'required|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => $validator->errors()->first()], 422);
        }

        $ad = Ad::findOrFail($id);
        
        if ($ad->status === 'Rejected') {
            return response()->json(['success' => false, 'message' => 'Ad is already rejected!']);
        }

        $ad->update([
            'status' => 'Rejected',
            'rejection_reason' => $request->rejection_reason,
            'rejected_at' => now(),
            'rejected_by' => auth()->id(),
        ]);

        // Send rejection notification to user
        Notification::create([
            'user_id' => $ad->user_id,
            'type' => 'ad_rejected',
            'title' => 'Your Ad Has Been Rejected',
            'message' => "Your ad '{$ad->title}' has been rejected. Reason: {$request->rejection_reason}",
            'data' => [
                'ad_id' => $ad->id,
                'ad_title' => $ad->title,
                'rejection_reason' => $request->rejection_reason
            ]
        ]);

        return response()->json(['success' => true, 'message' => 'Ad rejected successfully!']);
    }

    /**
     * Deactivate an ad.
     */
    public function deactivate($id)
    {
        $ad = Ad::findOrFail($id);
        
        if ($ad->status !== 'Active') {
            return response()->json(['success' => false, 'message' => 'Only active ads can be deactivated!']);
        }

        $ad->update([
            'status' => 'Pending',
            'deactivated_at' => now(),
            'deactivated_by' => auth()->id(),
        ]);

        return response()->json(['success' => true, 'message' => 'Ad deactivated successfully!']);
    }

    /**
     * Activate an ad.
     */
    public function activate($id)
    {
        $ad = Ad::findOrFail($id);
        
        if ($ad->status === 'Active') {
            return response()->json(['success' => false, 'message' => 'Ad is already active!']);
        }

        $ad->update([
            'status' => 'Active',
            'approved_at' => now(),
            'approved_by' => auth()->id(),
        ]);

        return response()->json(['success' => true, 'message' => 'Ad activated successfully!']);
    }

    /**
     * Bulk approve ads.
     */
    public function bulkApprove(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'ad_ids' => 'required|array',
            'ad_ids.*' => 'exists:ads,id',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => $validator->errors()->first()], 422);
        }

        $count = 0;
        foreach ($request->ad_ids as $adId) {
            $ad = Ad::find($adId);
            if ($ad && $ad->status !== 'Active') {
                $ad->update([
                    'status' => 'Active',
                    'approved_at' => now(),
                    'approved_by' => auth()->id(),
                ]);
                $count++;
            }
        }

        return response()->json(['success' => true, 'message' => "Successfully approved {$count} ads!"]);
    }

    /**
     * Bulk reject ads.
     */
    public function bulkReject(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'ad_ids' => 'required|array',
            'ad_ids.*' => 'exists:ads,id',
            'rejection_reason' => 'required|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => $validator->errors()->first()], 422);
        }

        $count = 0;
        foreach ($request->ad_ids as $adId) {
            $ad = Ad::find($adId);
            if ($ad && $ad->status !== 'Rejected') {
                $ad->update([
                    'status' => 'Rejected',
                    'rejection_reason' => $request->rejection_reason,
                    'rejected_at' => now(),
                    'rejected_by' => auth()->id(),
                ]);
                $count++;
            }
        }

        return response()->json(['success' => true, 'message' => "Successfully rejected {$count} ads!"]);
    }

    /**
     * Bulk deactivate ads.
     */
    public function bulkDeactivate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'ad_ids' => 'required|array',
            'ad_ids.*' => 'exists:ads,id',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => $validator->errors()->first()], 422);
        }

        $count = 0;
        foreach ($request->ad_ids as $adId) {
            $ad = Ad::find($adId);
            if ($ad && $ad->status === 'Active') {
                $ad->update([
                    'status' => 'Pending',
                    'deactivated_at' => now(),
                    'deactivated_by' => auth()->id(),
                ]);
                $count++;
            }
        }

        return response()->json(['success' => true, 'message' => "Successfully deactivated {$count} ads!"]);
    }

    /**
     * Show ad details.
     */
    public function show($id)
    {
        $ad = Ad::with(['user', 'category', 'subcategory', 'images', 'analytics'])
                 ->findOrFail($id);

        return view('admin.ads.show', compact('ad'));
    }

    /**
     * Edit ad.
     */
    public function edit($id)
    {
        $ad = Ad::with(['user', 'category', 'subcategory', 'images'])
                 ->findOrFail($id);

        return view('admin.ads.edit', compact('ad'));
    }

    /**
     * Update ad.
     */
    public function update(Request $request, $id)
    {
        $ad = Ad::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'price' => 'required|numeric|min:0',
            'currency' => 'required|string|max:10',
            'category_id' => 'required|exists:categories,id',
            'subcategory_id' => 'nullable|exists:subcategories,id',
            'status' => 'required|in:Active,Pending,Expired,Sold,Rejected,Draft',
            'country' => 'required|string|max:100',
            'state' => 'required|string|max:100',
            'city' => 'required|string|max:100',
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
            'attributes' => 'nullable|array',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $ad->update($request->all());

        // Update custom attributes
        if ($request->has('attributes')) {
            $ad->attributes()->delete();
             foreach ($request->attributes as $fieldId => $value) {
                if (is_array($value)) {
                    $value = json_encode($value);
                }
                
                AdAttributeValue::create([
                    'ad_id' => $ad->id,
                    'category_field_id' => $fieldId,
                    'value' => $value,
                ]);
            }
        }

        return redirect()->route('admin.ads.index')->with('success', 'Ad updated successfully!');
    }

    /**
     * Delete ad.
     */
    public function destroy($id)
    {
        $ad = Ad::findOrFail($id);
        $ad->delete();

        return response()->json(['success' => true, 'message' => 'Ad deleted successfully!']);
    }

    /**
     * Toggle featured status.
     */
    public function toggleFeatured($id)
    {
        $ad = Ad::findOrFail($id);
        $ad->update(['is_featured' => !$ad->is_featured]);

        $status = $ad->is_featured ? 'featured' : 'unfeatured';
        return response()->json(['success' => true, 'message' => "Ad {$status} successfully!"]);
    }

    /**
     * Toggle boost status.
     */
    public function toggleBoost($id)
    {
        $ad = Ad::findOrFail($id);
        $ad->update(['is_boosted' => !$ad->is_boosted]);

        $status = $ad->is_boosted ? 'boosted' : 'unboosted';
        return response()->json(['success' => true, 'message' => "Ad {$status} successfully!"]);
    }

    /**
     * Get approval settings.
     */
    public function getApprovalSettings()
    {
        $autoApprove = Setting::where('key', 'auto_approve_ads')->first();
        $defaultValidity = Setting::where('key', 'default_ad_validity_days')->first();
        
        return response()->json([
            'auto_approve_ads' => $autoApprove ? $autoApprove->value : true,
            'default_ad_validity_days' => $defaultValidity ? $defaultValidity->value : 30,
        ]);
    }

    /**
     * Update approval settings.
     */
    public function updateApprovalSettings(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'auto_approve_ads' => 'required|boolean',
            'default_ad_validity_days' => 'required|integer|min:1|max:365',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator);
        }

        Setting::updateOrCreate(
            ['key' => 'auto_approve_ads'],
            ['value' => $request->auto_approve_ads]
        );

        Setting::updateOrCreate(
            ['key' => 'default_ad_validity_days'],
            ['value' => $request->default_ad_validity_days]
        );

        return redirect()->back()->with('success', 'Approval settings updated successfully!');
    }

    /**
     * Get statistics for dashboard charts.
     */
    public function stats(Request $request)
    {
        $stats = [
            'total' => Ad::count(),
            'active' => Ad::where('status', 'Active')->count(),
            'pending' => Ad::where('status', 'Pending')->count(),
            'expired' => Ad::where('status', 'Expired')->count(),
            'sold' => Ad::where('status', 'Sold')->count(),
            'featured' => Ad::where('is_featured', true)->count(),
            'countries' => Ad::select('location_country')->distinct()->whereNotNull('location_country')->count(),
            'states' => Ad::select('location_state')->distinct()->whereNotNull('location_state')->count(),
        ];

        return response()->json($stats);
    }

    /**
     * Get chart data.
     */
    public function charts(Request $request)
    {
        // Status Distribution
        $statusDistribution = Ad::select('status', \DB::raw('count(*) as count'))
            ->groupBy('status')
            ->get();

        // Category Distribution
        $categoryDistribution = Ad::select('category_id', \DB::raw('count(*) as count'))
            ->whereNotNull('category_id')
            ->groupBy('category_id')
            ->get();

        // Location Distribution
        $locationDistribution = Ad::select('location_country as country', \DB::raw('count(*) as count'))
            ->whereNotNull('location_country')
            ->groupBy('location_country')
            ->get();
            
        // Featured Distribution
        $featuredDistribution = Ad::select('is_featured', \DB::raw('count(*) as count'))
            ->groupBy('is_featured')
            ->get();
            
        // Top Locations
        $topLocations = Ad::select('location_city as name', \DB::raw('count(*) as count'))
            ->whereNotNull('location_city')
            ->groupBy('location_city')
            ->orderBy('count', 'desc')
            ->limit(10)
            ->get();

        return response()->json([
            'status_distribution' => $statusDistribution,
            'category_distribution' => $categoryDistribution,
            'location_distribution' => $locationDistribution,
            'featured_distribution' => $featuredDistribution,
            'top_locations' => $topLocations,
        ]);
    }
}
