<?php

namespace App\Http\Controllers;

use App\Models\Ad;
use App\Models\AdImage;
use App\Models\AdAnalytics;
use App\Models\AdPostStatistics;
use App\Models\AdAttributeValue;
use App\Models\Category;
use App\Models\Subcategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class AdController extends Controller
{
    /**
     * Display a listing of ads.
     */
    public function index(Request $request)
    {
        $query = Ad::with(['user', 'category', 'subcategory', 'images'])
                    ->active()
                    ->orderByInterestAndLocation(
                        $request->latitude,
                        $request->longitude
                    );

        // Filters
        if ($request->filled('category')) {
            $query->byCategory($request->category);
        }

        if ($request->filled('subcategory')) {
            $query->bySubcategory($request->subcategory);
        }

        if ($request->filled('location')) {
            // For location-based search, we might need to implement geocoding
            // For now, we'll use the radius search if coordinates are provided
        }

        if ($request->filled('min_price') && $request->filled('max_price')) {
            $query->byPriceRange($request->min_price, $request->max_price);
        }

        if ($request->filled('latitude') && $request->filled('longitude')) {
            $query->nearLocation($request->latitude, $request->longitude, $request->radius ?? 50);
        }

        $ads = $query->paginate($request->per_page ?? 20);

        return response()->json([
            'success' => true,
            'data' => $ads,
        ]);
    }

    /**
     * Display the specified ad.
     */
    public function show($id)
    {
        $ad = Ad::with(['user', 'category', 'subcategory', 'images'])->findOrFail($id);

        // Increment impression count
        $ad->incrementImpressions();

        return response()->json([
            'success' => true,
            'data' => $ad,
        ]);
    }

    /**
     * Store a newly created ad.
     */
    public function store(Request $request)
    {
        $user = Auth::user();

        // Check if user can post ads
        if (!$user->canPostAd()) {
            return response()->json([
                'success' => false,
                'message' => 'You cannot post ads with your current plan',
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'price' => 'required|numeric|min:0',
            'currency' => 'required|string|max:10',
            'category_id' => 'required|exists:categories,id',
            'subcategory_id' => 'nullable|exists:subcategories,id',
            'location_name' => 'required|string',
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
            'images' => 'required|array|min:1',
            'images' => 'required|array|min:1',
            'images.*' => 'string',
            'attributes' => 'nullable|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        $ad = Ad::create([
            'user_id' => $user->id,
            'category_id' => $request->category_id,
            'subcategory_id' => $request->subcategory_id,
            'title' => $request->title,
            'description' => $request->description,
            'price' => $request->price,
            'currency' => $request->currency,
            'location_name' => $request->location_name,
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
            'status' => 'Active', // Default to active
        ]);

        // Create images
        foreach ($request->images as $index => $imageUrl) {
            AdImage::create([
                'ad_id' => $ad->id,
                'image_url' => $imageUrl,
                'order' => $index,
            ]);
        }

        // Create post statistics
        AdPostStatistics::create([
            'ad_id' => $ad->id,
            'total_views' => 0,
        ]);

        // Store custom attributes
        if ($request->has('attributes')) {
            foreach ($request->attributes as $fieldId => $value) {
                if (is_array($value)) {
                    $value = json_encode($value);
                }
                
                AdAttributeValue::create([
                    'ad_id' => $ad->id,
                    'category_field_id' => $fieldId,
                    'value' => $value,
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Ad created successfully',
            'data' => $ad->load(['category', 'subcategory', 'images']),
        ], 201);
    }

    /**
     * Update the specified ad.
     */
    public function update(Request $request, $id)
    {
        $ad = Ad::findOrFail($id);

        // Only allow ad owner to update
        if ($ad->user_id !== Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized',
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'title' => 'sometimes|required|string|max:255',
            'description' => 'sometimes|required|string',
            'price' => 'sometimes|required|numeric|min:0',
            'currency' => 'sometimes|required|string|max:10',
            'category_id' => 'sometimes|required|exists:categories,id',
            'subcategory_id' => 'sometimes|nullable|exists:subcategories,id',
            'location_name' => 'sometimes|required|string',
            'latitude' => 'sometimes|required|numeric',
            'longitude' => 'sometimes|required|numeric',
            'status' => 'sometimes|required|in:Active,Pending,Expired,Sold,Draft',
            'attributes' => 'nullable|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        $ad->update($request->all());

        // Update custom attributes
        if ($request->has('attributes')) {
            // Remove old attributes to avoid duplicates/stale data (simplest approach)
            // Or update existing ones. Deleting and recreating is easier for full sync.
            $ad->attributes()->delete();
            
            foreach ($request->attributes as $fieldId => $value) {
                 if (is_array($value)) {
                    $value = json_encode($value);
                }
                
                AdAttributeValue::create([
                    'ad_id' => $ad->id,
                    'category_field_id' => $fieldId,
                    'value' => $value,
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Ad updated successfully',
            'data' => $ad->fresh(['category', 'subcategory', 'images']),
        ]);
    }

    /**
     * Remove the specified ad.
     */
    public function destroy($id)
    {
        $ad = Ad::findOrFail($id);

        // Only allow ad owner to delete
        if ($ad->user_id !== Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized',
            ], 403);
        }

        $ad->delete();

        return response()->json([
            'success' => true,
            'message' => 'Ad deleted successfully',
        ]);
    }

    /**
     * Get ads for a specific user.
     */
    public function getUserAds($id)
    {
        $user = Auth::user();

        // Only allow users to view their own ads
        if ($user->id !== $id) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized',
            ], 403);
        }

        $ads = Ad::with(['category', 'subcategory', 'images'])
                    ->where('user_id', $id)
                    ->orderBy('created_at', 'desc')
                    ->paginate(20);

        return response()->json([
            'success' => true,
            'data' => $ads,
        ]);
    }

    /**
     * Boost an ad.
     */
    public function boostAd(Request $request, $id)
    {
        $ad = Ad::findOrFail($id);

        // Only allow ad owner to boost
        if ($ad->user_id !== Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized',
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'package_id' => 'required|exists:ad_packages,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Here you would typically handle payment and package logic
        // For now, just mark as boosted
        $ad->update([
            'is_boosted' => true,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Ad boosted successfully',
            'data' => $ad,
        ]);
    }

    /**
     * Feature an ad.
     */
    public function featureAd(Request $request, $id)
    {
        $ad = Ad::findOrFail($id);

        // Only allow ad owner to feature
        if ($ad->user_id !== Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized',
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'package_id' => 'required|exists:ad_packages,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Here you would typically handle payment and package logic
        // For now, just mark as featured
        $ad->update([
            'is_featured' => true,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Ad featured successfully',
            'data' => $ad,
        ]);
    }

    /**
     * Bump an ad.
     */
    public function bumpAd($id)
    {
        $ad = Ad::findOrFail($id);

        // Only allow ad owner to bump
        if ($ad->user_id !== Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized',
            ], 403);
        }

        if (!$ad->canBump()) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot bump ad at this time',
            ], 400);
        }

        $ad->bump();

        return response()->json([
            'success' => true,
            'message' => 'Ad bumped successfully',
            'data' => $ad,
        ]);
    }

    /**
     * Track ad view.
     */
    public function trackView($id)
    {
        $ad = Ad::findOrFail($id);

        // Increment view count
        $ad->incrementViews();

        // Create analytics record
        AdAnalytics::create([
            'ad_id' => $ad->id,
            'user_id' => Auth::id(),
            'event_type' => 'View',
            'ip_address' => $request->ip(),
            'device_info' => $request->header('User-Agent'),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'View tracked successfully',
        ]);
    }

    /**
     * Track ad click.
     */
    public function trackClick($id)
    {
        $ad = Ad::findOrFail($id);

        // Increment click count
        $ad->incrementClicks();

        // Create analytics record
        AdAnalytics::create([
            'ad_id' => $ad->id,
            'user_id' => Auth::id(),
            'event_type' => 'Click',
            'ip_address' => $request->ip(),
            'device_info' => $request->header('User-Agent'),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Click tracked successfully',
        ]);
    }
}
