<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class QueueMonitor extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'queue:monitor
                            {--check-failed : Check for failed jobs}
                            {--check-processing : Check for stuck jobs}
                            {--check-length : Check queue length}
                            {--alert-threshold=100 : Alert threshold for queue length}
                            {--stuck-threshold=300 : Stuck job threshold in seconds}
                            {--cleanup-stuck : Clean up stuck jobs}
                            {--cleanup-failed : Clean up old failed jobs}
                            {--cleanup-days=7 : Days to keep failed jobs}
                            {--json : Output in JSON format}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Monitor queue health and performance';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔍 Monitoring Queue Health...');
        $this->line('');

        $results = [
            'timestamp' => now()->toISOString(),
            'checks' => []
        ];

        // Check failed jobs
        if ($this->option('check-failed') || !$this->hasAnyOptions()) {
            $failedResults = $this->checkFailedJobs();
            $results['checks']['failed_jobs'] = $failedResults;
            
            if (!$this->option('json')) {
                $this->displayFailedJobs($failedResults);
            }
        }

        // Check processing jobs
        if ($this->option('check-processing') || !$this->hasAnyOptions()) {
            $processingResults = $this->checkProcessingJobs();
            $results['checks']['processing_jobs'] = $processingResults;
            
            if (!$this->option('json')) {
                $this->displayProcessingJobs($processingResults);
            }
        }

        // Check queue length
        if ($this->option('check-length') || !$this->hasAnyOptions()) {
            $lengthResults = $this->checkQueueLength();
            $results['checks']['queue_length'] = $lengthResults;
            
            if (!$this->option('json')) {
                $this->displayQueueLength($lengthResults);
            }
        }

        // Cleanup operations
        if ($this->option('cleanup-stuck')) {
            $this->cleanupStuckJobs();
        }

        if ($this->option('cleanup-failed')) {
            $this->cleanupFailedJobs();
        }

        // Output results
        if ($this->option('json')) {
            $this->line(json_encode($results, JSON_PRETTY_PRINT));
        } else {
            $this->displaySummary($results);
        }

        return 0;
    }

    /**
     * Check for failed jobs
     */
    protected function checkFailedJobs()
    {
        try {
            $failedJobs = DB::table('failed_jobs')
                ->select('queue', DB::raw('COUNT(*) as count'))
                ->groupBy('queue')
                ->get();

            $totalFailed = DB::table('failed_jobs')->count();
            $recentFailed = DB::table('failed_jobs')
                ->where('failed_at', '>=', now()->subHours(24))
                ->count();

            return [
                'total_failed' => $totalFailed,
                'recent_failed' => $recentFailed,
                'by_queue' => $failedJobs->toArray(),
                'status' => $totalFailed > 0 ? 'warning' : 'ok'
            ];
        } catch (\Exception $e) {
            Log::error('Queue monitor failed jobs check failed: ' . $e->getMessage());
            return [
                'error' => $e->getMessage(),
                'status' => 'error'
            ];
        }
    }

    /**
     * Check for stuck processing jobs
     */
    protected function checkProcessingJobs()
    {
        try {
            $stuckThreshold = $this->option('stuck-threshold');
            $stuckJobs = DB::table('jobs')
                ->where('reserved_at', '<=', now()->subSeconds($stuckThreshold))
                ->where('attempts', '>', 0)
                ->get();

            $processingJobs = DB::table('jobs')
                ->whereNotNull('reserved_at')
                ->count();

            return [
                'processing_jobs' => $processingJobs,
                'stuck_jobs' => $stuckJobs->count(),
                'stuck_job_details' => $stuckJobs->toArray(),
                'status' => $stuckJobs->count() > 0 ? 'warning' : 'ok'
            ];
        } catch (\Exception $e) {
            Log::error('Queue monitor processing jobs check failed: ' . $e->getMessage());
            return [
                'error' => $e->getMessage(),
                'status' => 'error'
            ];
        }
    }

    /**
     * Check queue length
     */
    protected function checkQueueLength()
    {
        try {
            $alertThreshold = $this->option('alert-threshold');
            
            $queueStats = DB::table('jobs')
                ->select('queue', DB::raw('COUNT(*) as pending'))
                ->whereNull('reserved_at')
                ->groupBy('queue')
                ->get();

            $totalPending = DB::table('jobs')
                ->whereNull('reserved_at')
                ->count();

            return [
                'total_pending' => $totalPending,
                'by_queue' => $queueStats->toArray(),
                'alert_threshold' => $alertThreshold,
                'status' => $totalPending > $alertThreshold ? 'alert' : 'ok'
            ];
        } catch (\Exception $e) {
            Log::error('Queue monitor queue length check failed: ' . $e->getMessage());
            return [
                'error' => $e->getMessage(),
                'status' => 'error'
            ];
        }
    }

    /**
     * Clean up stuck jobs
     */
    protected function cleanupStuckJobs()
    {
        $this->info('🧹 Cleaning up stuck jobs...');
        
        try {
            $stuckThreshold = $this->option('stuck-threshold');
            $stuckJobs = DB::table('jobs')
                ->where('reserved_at', '<=', now()->subSeconds($stuckThreshold))
                ->where('attempts', '>', 0)
                ->get();

            if ($stuckJobs->count() > 0) {
                $deleted = DB::table('jobs')
                    ->where('reserved_at', '<=', now()->subSeconds($stuckThreshold))
                    ->where('attempts', '>', 0)
                    ->delete();

                $this->info("✅ Cleaned up {$deleted} stuck jobs");
                Log::info("Queue monitor cleaned up {$deleted} stuck jobs");
            } else {
                $this->info('✅ No stuck jobs found');
            }
        } catch (\Exception $e) {
            $this->error('❌ Failed to cleanup stuck jobs: ' . $e->getMessage());
            Log::error('Queue monitor cleanup stuck jobs failed: ' . $e->getMessage());
        }
    }

    /**
     * Clean up old failed jobs
     */
    protected function cleanupFailedJobs()
    {
        $this->info('🧹 Cleaning up old failed jobs...');
        
        try {
            $daysToKeep = $this->option('cleanup-days');
            $cutoffDate = now()->subDays($daysToKeep);
            
            $deleted = DB::table('failed_jobs')
                ->where('failed_at', '<=', $cutoffDate)
                ->delete();

            $this->info("✅ Cleaned up {$deleted} old failed jobs");
            Log::info("Queue monitor cleaned up {$deleted} failed jobs older than {$daysToKeep} days");
        } catch (\Exception $e) {
            $this->error('❌ Failed to cleanup failed jobs: ' . $e->getMessage());
            Log::error('Queue monitor cleanup failed jobs failed: ' . $e->getMessage());
        }
    }

    /**
     * Display failed jobs results
     */
    protected function displayFailedJobs($results)
    {
        $this->section('Failed Jobs');
        
        if (isset($results['error'])) {
            $this->error('❌ Failed to check failed jobs: ' . $results['error']);
            return;
        }

        $this->info("Total Failed Jobs: {$results['total_failed']}");
        $this->info("Recent Failed Jobs (24h): {$results['recent_failed']}");
        
        if (!empty($results['by_queue'])) {
            $this->table(
                ['Queue', 'Count'],
                array_map(fn($item) => [$item->queue, $item->count], $results['by_queue'])
            );
        }
        
        $this->line('');
    }

    /**
     * Display processing jobs results
     */
    protected function displayProcessingJobs($results)
    {
        $this->section('Processing Jobs');
        
        if (isset($results['error'])) {
            $this->error('❌ Failed to check processing jobs: ' . $results['error']);
            return;
        }

        $this->info("Processing Jobs: {$results['processing_jobs']}");
        $this->info("Stuck Jobs: {$results['stuck_jobs']}");
        
        if ($results['stuck_jobs'] > 0) {
            $this->warn('⚠️  Found stuck jobs!');
            if (!empty($results['stuck_job_details'])) {
                $this->table(
                    ['Job ID', 'Queue', 'Attempts', 'Reserved At'],
                    array_map(fn($item) => [
                        $item->id, 
                        $item->queue, 
                        $item->attempts, 
                        $item->reserved_at
                    ], $results['stuck_job_details'])
                );
            }
        }
        
        $this->line('');
    }

    /**
     * Display queue length results
     */
    protected function displayQueueLength($results)
    {
        $this->section('Queue Length');
        
        if (isset($results['error'])) {
            $this->error('❌ Failed to check queue length: ' . $results['error']);
            return;
        }

        $this->info("Total Pending Jobs: {$results['total_pending']}");
        $this->info("Alert Threshold: {$results['alert_threshold']}");
        
        if (!empty($results['by_queue'])) {
            $this->table(
                ['Queue', 'Pending'],
                array_map(fn($item) => [$item->queue, $item->pending], $results['by_queue'])
            );
        }
        
        if ($results['total_pending'] > $results['alert_threshold']) {
            $this->warn('⚠️  Queue length exceeds alert threshold!');
        }
        
        $this->line('');
    }

    /**
     * Display summary
     */
    protected function displaySummary($results)
    {
        $this->section('Queue Health Summary');
        
        $status = 'ok';
        $issues = [];
        
        foreach ($results['checks'] as $check => $data) {
            if ($data['status'] === 'error') {
                $status = 'error';
                $issues[] = "❌ {$check}: Error occurred";
            } elseif ($data['status'] === 'warning') {
                $status = 'warning';
                $issues[] = "⚠️  {$check}: Issues detected";
            } elseif ($data['status'] === 'alert') {
                $status = 'alert';
                $issues[] = "🚨 {$check}: Alert threshold exceeded";
            }
        }

        if ($status === 'ok') {
            $this->info('✅ All queue health checks passed!');
        } else {
            $this->warn('⚠️  Queue health issues detected:');
            foreach ($issues as $issue) {
                $this->line($issue);
            }
        }
        
        $this->line('');
        $this->info("Timestamp: {$results['timestamp']}");
    }

    /**
     * Check if any specific options were provided
     */
    protected function hasAnyOptions()
    {
        return $this->option('check-failed') ||
               $this->option('check-processing') ||
               $this->option('check-length') ||
               $this->option('cleanup-stuck') ||
               $this->option('cleanup-failed');
    }
}
