# Queue Worker Setup Guide for CLX Pro

This guide will help you configure queue workers for background processing of scheduled notifications in CLX Pro.

## 🚀 Quick Start

### 1. Configure Queue Driver

Set up your queue connection in `.env`:

```env
# Queue Configuration
QUEUE_CONNECTION=database

# For Redis (recommended for production)
# QUEUE_CONNECTION=redis
# REDIS_HOST=127.0.0.1
# REDIS_PASSWORD=null
# REDIS_PORT=6379
```

### 2. Create Queue Tables

Run the migration to create queue tables:

```bash
php artisan queue:table
php artisan migrate
```

### 3. Start Queue Worker

Start the queue worker for processing notifications:

```bash
# Basic queue worker
php artisan queue:work

# Queue worker with specific connection
php artisan queue:work --queue=notifications

# Queue worker with daemon mode (production)
php artisan queue:work --daemon --queue=notifications --tries=3
```

## 🔧 Detailed Configuration

### Queue Driver Selection

#### Database Driver (Development/Testing)
- **Pros**: Simple setup, no additional services required
- **Cons**: Slower, not suitable for high volume
- **Use Case**: Development, testing, low-volume production

#### Redis Driver (Production Recommended)
- **Pros**: Fast, reliable, supports multiple workers
- **Cons**: Requires Redis server
- **Use Case**: Production environments, high-volume processing

#### SQS Driver (Cloud)
- **Pros**: Managed service, scalable
- **Cons**: AWS costs, network dependency
- **Use Case**: AWS-based deployments

### Queue Configuration

#### Environment Variables

```env
# Queue Connection
QUEUE_CONNECTION=database

# Redis Configuration (if using Redis)
REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379
REDIS_DB=0

# Queue Settings
QUEUE_BATCH_SIZE=100
QUEUE_RETRY_ATTEMPTS=3
QUEUE_RETRY_DELAY=60
```

#### Queue Worker Configuration

```php
// config/queue.php
'connections' => [
    'database' => [
        'driver' => 'database',
        'table' => 'jobs',
        'queue' => 'default',
        'retry_after' => 90,
        'after_commit' => false,
    ],
    
    'redis' => [
        'driver' => 'redis',
        'connection' => 'default',
        'queue' => env('REDIS_QUEUE', 'default'),
        'retry_after' => 90,
        'block_for' => null,
    ],
],
```

### Queue Worker Management

#### Supervisor Configuration (Production)

Create a supervisor configuration file:

```ini
; /etc/supervisor/conf.d/clxpro-worker.conf
[program:clxpro-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /path/to/clx_backend/artisan queue:work --queue=notifications --sleep=3 --tries=3 --daemon
autostart=true
autorestart=true
user=www-data
numprocs=4
redirect_stderr=true
stdout_logfile=/path/to/clx_backend/storage/logs/worker.log
stopwaitsecs=3600
```

#### Supervisor Commands

```bash
# Reload supervisor configuration
sudo supervisorctl reread
sudo supervisorctl update

# Start queue workers
sudo supervisorctl start clxpro-worker:*

# Check worker status
sudo supervisorctl status

# Restart workers
sudo supervisorctl restart clxpro-worker:*

# Stop workers
sudo supervisorctl stop clxpro-worker:*
```

### Queue Monitoring

#### Queue Statistics

Monitor queue performance with Artisan commands:

```bash
# Check queue status
php artisan queue:work --status

# List failed jobs
php artisan queue:failed

# Retry failed jobs
php artisan queue:retry all

# Flush failed jobs
php artisan queue:flush
```

#### Queue Dashboard

Install Laravel Horizon for advanced queue monitoring:

```bash
composer require laravel/horizon
php artisan horizon:install
php artisan migrate
```

### Queue Job Configuration

#### Notification Job Configuration

```php
// app/Jobs/SendScheduledNotification.php
class SendScheduledNotification implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $tries = 3;
    public $maxExceptions = 3;
    public $backoff = 60;

    public function __construct(
        public Notification $notification
    ) {}

    public function handle()
    {
        // Send notification logic
        $this->notification->send();
    }

    public function failed(\Throwable $exception)
    {
        // Handle job failure
        Log::error('Notification job failed', [
            'notification_id' => $this->notification->id,
            'exception' => $exception->getMessage()
        ]);
    }
}
```

#### Queue Priorities

Configure queue priorities for different notification types:

```php
// Send high-priority notifications first
SendScheduledNotification::dispatch($notification)
    ->onQueue('high_priority')
    ->delay(now()->addMinutes(1));

// Send regular notifications
SendScheduledNotification::dispatch($notification)
    ->onQueue('default')
    ->delay(now()->addMinutes(5));
```

## 🚨 Troubleshooting

### Common Issues

#### 1. Queue Worker Not Processing Jobs

**Cause**: Worker not started or configuration issues
**Solution**:
```bash
# Check if worker is running
ps aux | grep queue:work

# Start worker manually
php artisan queue:work --verbose

# Check logs for errors
tail -f storage/logs/laravel.log
```

#### 2. Jobs Stuck in Processing

**Cause**: Job taking too long or failing repeatedly
**Solution**:
```bash
# Check processing jobs
php artisan queue:work --status

# Clear stuck jobs
php artisan queue:restart

# Adjust retry settings
php artisan queue:work --tries=5 --delay=120
```

#### 3. Memory Issues

**Cause**: Worker consuming too much memory
**Solution**:
```bash
# Restart worker after each job
php artisan queue:work --once

# Limit memory usage
php artisan queue:work --memory=128

# Use supervisor with memory limits
```

#### 4. Database Connection Issues

**Cause**: Database connection lost during long-running worker
**Solution**:
```bash
# Restart worker periodically
php artisan queue:work --max-jobs=1000

# Use connection pooling
DB_CONNECTION_POOL_SIZE=10
```

### Performance Optimization

#### Worker Optimization

```bash
# Optimize worker performance
php artisan queue:work \
    --queue=notifications \
    --sleep=1 \
    --tries=3 \
    --memory=256 \
    --timeout=300 \
    --daemon
```

#### Queue Optimization

```php
// config/queue.php
'connections' => [
    'database' => [
        'driver' => 'database',
        'table' => 'jobs',
        'queue' => 'default',
        'retry_after' => 90,
        'after_commit' => false,
        'expire' => 604800, // 7 days
    ],
],
```

#### Monitoring Setup

```bash
# Set up monitoring
composer require spatie/laravel-health

# Configure health checks
php artisan vendor:publish --provider="Spatie\Health\HealthServiceProvider"
```

## 📊 Queue Best Practices

### 1. Job Design

- **Keep jobs small and focused**
- **Handle exceptions gracefully**
- **Use appropriate retry logic**
- **Log important events**

### 2. Worker Management

- **Use supervisor for production**
- **Monitor worker health**
- **Set appropriate memory limits**
- **Restart workers periodically**

### 3. Queue Monitoring

- **Monitor queue length**
- **Track job processing time**
- **Set up alerts for failures**
- **Review failed jobs regularly**

### 4. Performance

- **Use Redis for high volume**
- **Optimize database queries**
- **Use queue priorities**
- **Batch similar jobs**

## 🚀 Production Deployment

### 1. Environment Setup

```bash
# Install Redis (Ubuntu)
sudo apt update
sudo apt install redis-server

# Configure Redis
sudo systemctl enable redis-server
sudo systemctl start redis-server
```

### 2. Supervisor Setup

```bash
# Install supervisor
sudo apt install supervisor

# Create worker configuration
sudo nano /etc/supervisor/conf.d/clxpro-worker.conf
```

### 3. Queue Worker Deployment

```bash
# Deploy with queue workers
php artisan migrate
php artisan queue:restart
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start clxpro-worker:*
```

### 4. Monitoring Setup

```bash
# Set up monitoring
composer require spatie/laravel-health
php artisan health:install
php artisan health:check
```

## 📞 Support

For additional help:

1. **Laravel Queue Documentation**: [laravel.com/docs/queues](https://laravel.com/docs/queues)
2. **Supervisor Documentation**: [supervisord.org](http://supervisord.org/)
3. **Redis Documentation**: [redis.io/documentation](https://redis.io/documentation)

## 🎯 Next Steps

After completing queue worker setup:

1. **Test Queue Processing**: Send test notifications
2. **Monitor Performance**: Set up monitoring and alerts
3. **Scale Workers**: Adjust worker count based on load
4. **Optimize Performance**: Fine-tune configuration

---

**Note**: Always test queue worker configuration in a staging environment before deploying to production.
