# Production Deployment Guide for CLX Pro

This guide will help you deploy CLX Pro to a production environment with all optimizations and security measures.

## 🚀 Quick Start

### 1. Environment Preparation

```bash
# Set production environment
cp .env.example .env
# Edit .env with production values

# Install dependencies
composer install --optimize-autoloader --no-dev
npm install --production
```

### 2. Database Setup

```bash
# Run migrations
php artisan migrate --force

# Seed production data
php artisan db:seed --force
```

### 3. Asset Optimization

```bash
# Build production assets
npm run build

# Optimize Laravel
php artisan config:cache
php artisan route:cache
php artisan view:cache
php artisan event:cache
```

### 4. Queue Worker Setup

```bash
# Start queue workers
php artisan queue:work --daemon --queue=notifications --tries=3

# Set up supervisor (recommended)
sudo supervisorctl reread
sudo supervisorctl update
sudo supervisorctl start clxpro-worker:*
```

## 🔧 Detailed Deployment

### Environment Configuration

#### Production Environment Variables

```env
# Application
APP_ENV=production
APP_DEBUG=false
APP_URL=https://your-domain.com

# Database
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=clxpro_prod
DB_USERNAME=clxpro_user
DB_PASSWORD=secure_password

# Cache
CACHE_DRIVER=redis
SESSION_DRIVER=redis
QUEUE_CONNECTION=redis

# Redis
REDIS_HOST=127.0.0.1
REDIS_PASSWORD=your_redis_password
REDIS_PORT=6379

# Logging
LOG_CHANNEL=stack
LOG_LEVEL=error

# Security
APP_KEY=your-production-app-key
SESSION_SECURE_COOKIE=true
SESSION_SAME_SITE=strict

# Queue Settings
QUEUE_BATCH_SIZE=100
QUEUE_RETRY_ATTEMPTS=3
QUEUE_RETRY_DELAY=60

# Notification Settings
FIREBASE_PROJECT_ID=your-firebase-project
FIREBASE_SERVER_KEY=your-firebase-server-key
FIREBASE_API_KEY=your-firebase-api-key

# Mobile App Deep Links
DEEP_LINK_SCHEME=clxpro
DEEP_LINK_HOST=app
DEEP_LINK_BASE_URL=https://clxpro.page.link
```

### Web Server Configuration

#### Nginx Configuration

```nginx
server {
    listen 80;
    server_name your-domain.com;
    return 301 https://$server_name$request_uri;
}

server {
    listen 443 ssl http2;
    server_name your-domain.com;

    root /path/to/clx_backend/public;
    index index.php index.html index.htm;

    # SSL Configuration
    ssl_certificate /path/to/ssl/cert.pem;
    ssl_certificate_key /path/to/ssl/key.pem;
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_ciphers ECDHE-RSA-AES128-GCM-SHA256:ECDHE-RSA-AES256-GCM-SHA384;
    ssl_prefer_server_ciphers off;

    # Security Headers
    add_header X-Frame-Options DENY;
    add_header X-Content-Type-Options nosniff;
    add_header X-XSS-Protection "1; mode=block";
    add_header Referrer-Policy "strict-origin-when-cross-origin";
    add_header Content-Security-Policy "default-src 'self'; script-src 'self' 'unsafe-inline' 'unsafe-eval'; style-src 'self' 'unsafe-inline'; img-src 'self' data: https:; font-src 'self' https:;";

    # Gzip Compression
    gzip on;
    gzip_vary on;
    gzip_min_length 1024;
    gzip_types text/plain text/css text/xml text/javascript application/javascript application/xml+rss application/json;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.1-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }

    location ~ /\.ht {
        deny all;
    }

    # Cache static assets
    location ~* \.(js|css|png|jpg|jpeg|gif|ico|svg)$ {
        expires 1y;
        add_header Cache-Control "public, immutable";
    }
}
```

#### Apache Configuration

```apache
<VirtualHost *:80>
    ServerName your-domain.com
    DocumentRoot /path/to/clx_backend/public
    
    RewriteEngine On
    RewriteCond %{HTTPS} off
    RewriteRule ^(.*)$ https://%{HTTP_HOST}%{REQUEST_URI} [L,R=301]
    
    <Directory /path/to/clx_backend/public>
        AllowOverride All
        Require all granted
    </Directory>
</VirtualHost>

<VirtualHost *:443>
    ServerName your-domain.com
    DocumentRoot /path/to/clx_backend/public
    
    SSLEngine on
    SSLCertificateFile /path/to/ssl/cert.pem
    SSLCertificateKeyFile /path/to/ssl/key.pem
    
    <Directory /path/to/clx_backend/public>
        AllowOverride All
        Require all granted
    </Directory>
    
    # Security Headers
    Header always set X-Frame-Options DENY
    Header always set X-Content-Type-Options nosniff
    Header always set X-XSS-Protection "1; mode=block"
    Header always set Referrer-Policy "strict-origin-when-cross-origin"
</VirtualHost>
```

### Database Optimization

#### MySQL Configuration

```ini
# /etc/mysql/mysql.conf.d/mysqld.cnf
[mysqld]
# Performance Settings
innodb_buffer_pool_size = 1G
innodb_log_file_size = 256M
innodb_flush_method = O_DIRECT
innodb_flush_log_at_trx_commit = 2

# Connection Settings
max_connections = 200
thread_cache_size = 50

# Query Cache (MySQL 5.7 and below)
query_cache_type = 1
query_cache_size = 256M

# Other Settings
tmp_table_size = 64M
max_heap_table_size = 64M
```

#### Database Indexes

```sql
-- Optimize notification queries
CREATE INDEX idx_notifications_status_scheduled ON notifications(status, scheduled_at);
CREATE INDEX idx_notifications_target_type ON notifications(target_type);
CREATE INDEX idx_notification_deliveries_status ON notification_deliveries(status);

-- Optimize user queries
CREATE INDEX idx_users_status ON users(status);
CREATE INDEX idx_users_created_at ON users(created_at);

-- Optimize ad queries
CREATE INDEX idx_ads_status_created ON ads(status, created_at);
CREATE INDEX idx_ads_category_location ON ads(category_id, location);
```

### Queue Worker Optimization

#### Supervisor Configuration

```ini
; /etc/supervisor/conf.d/clxpro-worker.conf
[program:clxpro-worker]
process_name=%(program_name)s_%(process_num)02d
command=php /path/to/clx_backend/artisan queue:work --queue=notifications,default --sleep=3 --tries=3 --memory=256 --timeout=300 --daemon
autostart=true
autorestart=true
user=www-data
numprocs=4
redirect_stderr=true
stdout_logfile=/path/to/clx_backend/storage/logs/worker.log
stopwaitsecs=3600
priority=999

[program:clxpro-scheduler]
process_name=%(program_name)s
command=php /path/to/clx_backend/artisan schedule:work
autostart=true
autorestart=true
user=www-data
redirect_stderr=true
stdout_logfile=/path/to/clx_backend/storage/logs/scheduler.log
priority=998
```

#### Queue Monitoring

```bash
# Monitor queue health
php artisan queue:monitor --json

# Check worker status
sudo supervisorctl status

# View logs
tail -f /path/to/clx_backend/storage/logs/worker.log
```

### Caching Strategy

#### Redis Configuration

```ini
# /etc/redis/redis.conf
maxmemory 512mb
maxmemory-policy allkeys-lru
tcp-keepalive 300
timeout 0
```

#### Laravel Cache Configuration

```php
// config/cache.php
'redis' => [
    'driver' => 'redis',
    'connection' => 'cache',
    'prefix' => env('CACHE_PREFIX', 'clxpro_cache_'),
],

// config/session.php
'redis' => [
    'client' => 'phpredis',
    'cluster' => false,
    'options' => [
        'cluster' => 'redis',
        'prefix' => env('SESSION_PREFIX', 'clxpro_session_'),
    ],
    'default' => [
        'host' => env('REDIS_HOST', '127.0.0.1'),
        'password' => env('REDIS_PASSWORD', null),
        'port' => env('REDIS_PORT', 6379),
        'database' => 0,
    ],
],
```

### Security Hardening

#### Application Security

```php
// config/app.php
'cipher' => 'AES-256-CBC',
'key' => env('APP_KEY'),
'strict_mode' => true,

// config/session.php
'driver' => env('SESSION_DRIVER', 'redis'),
'encrypt' => true,
'http_only' => true,
'same_site' => 'strict',
'secure' => env('SESSION_SECURE_COOKIE', false),
```

#### Rate Limiting

```php
// app/Http/Kernel.php
protected $middlewareGroups = [
    'web' => [
        // ... other middleware
        \Illuminate\Routing\Middleware\ThrottleRequests::class . ':60,1',
    ],
];

// routes/web.php
Route::middleware(['throttle:1000,1'])->group(function () {
    // Your routes here
});
```

#### CORS Configuration

```php
// config/cors.php
return [
    'paths' => ['api/*'],
    'allowed_methods' => ['*'],
    'allowed_origins' => ['https://your-domain.com', 'https://app.clxpro.com'],
    'allowed_origins_patterns' => [],
    'allowed_headers' => ['*'],
    'exposed_headers' => [],
    'max_age' => 0,
    'supports_credentials' => false,
];
```

### Monitoring and Logging

#### Log Configuration

```php
// config/logging.php
'channels' => [
    'stack' => [
        'driver' => 'stack',
        'channels' => ['single', 'slack'],
        'ignore_exceptions' => false,
    ],
    
    'single' => [
        'driver' => 'single',
        'path' => storage_path('logs/laravel.log'),
        'level' => env('LOG_LEVEL', 'error'),
    ],
    
    'slack' => [
        'driver' => 'slack',
        'url' => env('LOG_SLACK_WEBHOOK_URL'),
        'username' => 'CLX Pro Bot',
        'emoji' => ':boom:',
        'level' => 'error',
    ],
],
```

#### Health Checks

```php
// routes/web.php
Route::get('/health', function () {
    return response()->json([
        'status' => 'healthy',
        'timestamp' => now()->toISOString(),
        'version' => config('app.version', '1.0.0'),
        'environment' => app()->environment(),
        'database' => DB::connection()->getPdo() ? 'connected' : 'disconnected',
        'cache' => Cache::getStore()->getRedis()->ping() === '+PONG' ? 'connected' : 'disconnected',
    ]);
});
```

### Performance Optimization

#### PHP-FPM Configuration

```ini
; /etc/php/8.1/fpm/pool.d/clxpro.conf
[clxpro]
user = www-data
group = www-data
listen = /run/php/php8.1-fpm-clxpro.sock
listen.owner = www-data
listen.group = www-data
pm = dynamic
pm.max_children = 50
pm.start_servers = 10
pm.min_spare_servers = 5
pm.max_spare_servers = 20
pm.max_requests = 500
```

#### OPcache Configuration

```ini
; /etc/php/8.1/fpm/conf.d/10-opcache.ini
opcache.enable=1
opcache.memory_consumption=256
opcache.interned_strings_buffer=16
opcache.max_accelerated_files=20000
opcache.validate_timestamps=0
opcache.save_comments=1
opcache.fast_shutdown=1
```

### Backup Strategy

#### Database Backup

```bash
#!/bin/bash
# /usr/local/bin/backup-clxpro.sh
BACKUP_DIR="/backup/clxpro"
DATE=$(date +%Y%m%d_%H%M%S)
DB_NAME="clxpro_prod"

mkdir -p $BACKUP_DIR
mysqldump -u root -p$DB_PASSWORD --single-transaction --routines --triggers $DB_NAME > $BACKUP_DIR/clxpro_$DATE.sql
gzip $BACKUP_DIR/clxpro_$DATE.sql

# Keep only last 7 days
find $BACKUP_DIR -name "*.gz" -mtime +7 -delete
```

#### Application Backup

```bash
#!/bin/bash
# /usr/local/bin/backup-clxpro-app.sh
BACKUP_DIR="/backup/clxpro-app"
DATE=$(date +%Y%m%d_%H%M%S)

# Backup application files
tar -czf $BACKUP_DIR/clxpro-app_$DATE.tar.gz /path/to/clx_backend

# Backup storage files
tar -czf $BACKUP_DIR/clxpro-storage_$DATE.tar.gz /path/to/clx_backend/storage

# Keep only last 7 days
find $BACKUP_DIR -name "*.tar.gz" -mtime +7 -delete
```

### Deployment Automation

#### Deploy Script

```bash
#!/bin/bash
# deploy.sh
set -e

echo "🚀 Starting CLX Pro deployment..."

# Variables
APP_PATH="/path/to/clx_backend"
BACKUP_PATH="/backup/deployments/clxpro_$(date +%Y%m%d_%H%M%S)"

echo "📦 Creating backup..."
mkdir -p $BACKUP_PATH
cp -r $APP_PATH $BACKUP_PATH

echo "📥 Pulling latest code..."
cd $APP_PATH
git pull origin main

echo "📦 Installing dependencies..."
composer install --optimize-autoloader --no-dev
npm install --production

echo "🔧 Running migrations..."
php artisan migrate --force

echo "🎨 Building assets..."
npm run build

echo "⚙️  Optimizing application..."
php artisan config:cache
php artisan route:cache
php artisan view:cache
php artisan event:cache

echo "🔄 Restarting services..."
sudo supervisorctl restart clxpro-worker:*
sudo systemctl reload nginx

echo "✅ Deployment completed successfully!"
echo "📍 Backup location: $BACKUP_PATH"
```

## 🚨 Troubleshooting

### Common Issues

#### 1. Queue Workers Not Processing

**Symptoms**: Jobs piling up in queue, notifications not sending
**Solution**:
```bash
# Check worker status
sudo supervisorctl status

# Restart workers
sudo supervisorctl restart clxpro-worker:*

# Check logs
tail -f storage/logs/worker.log
```

#### 2. High Memory Usage

**Symptoms**: Application slow, memory exhaustion
**Solution**:
```bash
# Optimize PHP-FPM
pm.max_children = 30
pm.max_requests = 200

# Clear cache regularly
php artisan cache:clear
php artisan view:clear
php artisan route:clear
```

#### 3. Database Connection Issues

**Symptoms**: Database timeouts, connection errors
**Solution**:
```sql
-- Check connection pool
SHOW PROCESSLIST;

-- Optimize MySQL settings
innodb_buffer_pool_size = 2G
max_connections = 300
```

#### 4. SSL Certificate Issues

**Symptoms**: HTTPS not working, certificate errors
**Solution**:
```bash
# Check certificate
openssl x509 -in /path/to/cert.pem -text -noout

# Renew certificate
sudo certbot renew --quiet
```

### Performance Monitoring

#### Key Metrics to Monitor

1. **Queue Length**: Keep queue length below 1000
2. **Response Time**: API response time under 500ms
3. **Memory Usage**: PHP-FPM memory under 256MB per process
4. **Database Connections**: MySQL connections under 80% of max
5. **Error Rate**: Application errors under 1%

#### Monitoring Tools

```bash
# System monitoring
htop
iotop
netstat -tulpn

# Application monitoring
php artisan queue:monitor
php artisan schedule:list
tail -f storage/logs/laravel.log

# Database monitoring
mysqladmin processlist
SHOW STATUS LIKE 'Threads_connected';
```

## 📞 Support

For additional help:

1. **Laravel Documentation**: [laravel.com/docs](https://laravel.com/docs)
2. **Server Configuration**: [nginx.org](https://nginx.org/)
3. **Database Optimization**: [mysql.com](https://mysql.com/)
4. **Queue Management**: [supervisord.org](http://supervisord.org/)

## 🎯 Post-Deployment Checklist

- [ ] Application accessible via HTTPS
- [ ] Queue workers running and processing jobs
- [ ] Database connections stable
- [ ] Cache working properly
- [ ] Logs being written correctly
- [ ] SSL certificate valid
- [ ] Performance within acceptable limits
- [ ] Backup systems operational
- [ ] Monitoring systems active
- [ ] Security measures in place

---

**Note**: Always test deployment procedures in a staging environment before applying to production.
