# Firebase Setup Guide for CLX Pro

This guide will help you configure Firebase for the CLX Pro notification system.

## 🚀 Quick Start

### 1. Create Firebase Project

1. Go to [Firebase Console](https://console.firebase.google.com/)
2. Click "Add project"
3. Name your project "CLX Pro Notifications"
4. Follow the setup wizard

### 2. Configure Firebase Cloud Messaging

1. In your Firebase project, go to **Project Settings** > **Cloud Messaging**
2. Copy the **Server key** and **Sender ID**
3. These will be used in your `.env` file

### 3. Create Service Account

1. Go to **Project Settings** > **Service Accounts**
2. Click "Generate new private key"
3. Download the JSON file
4. Extract the private key content for your `.env` file

### 4. Configure Environment

Copy the values from your Firebase project into your `.env` file:

```env
# Firebase Configuration
FIREBASE_PROJECT_ID=your-firebase-project-id
FIREBASE_CLIENT_EMAIL=your-service-account@your-project-id.iam.gserviceaccount.com
FIREBASE_PRIVATE_KEY="-----BEGIN PRIVATE KEY-----\nYOUR_PRIVATE_KEY_HERE\n-----END PRIVATE KEY-----\n"
FIREBASE_DATABASE_URL=https://your-project-id.firebaseio.com
FIREBASE_STORAGE_BUCKET=your-project-id.appspot.com
FIREBASE_SENDER_ID=your-sender-id
FIREBASE_APP_ID=your-app-id
FIREBASE_MEASUREMENT_ID=your-measurement-id

# Firebase Messaging
FIREBASE_SERVER_KEY=your-server-key-from-firebase-console
FIREBASE_API_KEY=your-api-key-from-firebase-console
```

## 🔧 Detailed Configuration

### Firebase Project Setup

#### Step 1: Enable Required Services

In your Firebase console, enable these services:
- **Firebase Cloud Messaging** (FCM)
- **Firebase Authentication** (for user management)
- **Firebase Realtime Database** (optional, for additional data)
- **Firebase Storage** (optional, for media files)

#### Step 2: Configure Authentication

1. Go to **Authentication** > **Sign-in method**
2. Enable **Email/Password** authentication
3. This allows Firebase to manage user tokens

#### Step 3: Set Up Mobile Apps

##### Android Configuration

1. Go to **Project Settings** > **General**
2. Click "Add app" > **Android**
3. Register your Android package name (e.g., `com.clxpro.app`)
4. Download `google-services.json`
5. Place it in your Flutter app's `android/app/` directory

##### iOS Configuration

1. Go to **Project Settings** > **General**
2. Click "Add app" > **iOS**
3. Register your iOS bundle ID (e.g., `com.clxpro.app`)
4. Download `GoogleService-Info.plist`
5. Add it to your Flutter iOS project

### Environment Variables Explained

| Variable | Description | Where to Find |
|----------|-------------|---------------|
| `FIREBASE_PROJECT_ID` | Your Firebase project ID | Project Settings > General |
| `FIREBASE_CLIENT_EMAIL` | Service account email | Project Settings > Service Accounts |
| `FIREBASE_PRIVATE_KEY` | Service account private key | Project Settings > Service Accounts |
| `FIREBASE_SERVER_KEY` | FCM server key | Project Settings > Cloud Messaging |
| `FIREBASE_API_KEY` | Firebase API key | Project Settings > General |
| `FIREBASE_DATABASE_URL` | Realtime Database URL | Database > Realtime Database |
| `FIREBASE_STORAGE_BUCKET` | Storage bucket URL | Storage > Files |
| `FIREBASE_SENDER_ID` | FCM sender ID | Project Settings > Cloud Messaging |
| `FIREBASE_APP_ID` | Firebase app ID | Project Settings > General |
| `FIREBASE_MEASUREMENT_ID` | Analytics measurement ID | Project Settings > Integrations > Analytics |

## 🧪 Testing Firebase Configuration

### Using Artisan Command

Test your Firebase configuration using the built-in command:

```bash
# Test all Firebase configuration
php artisan firebase:test --all

# Test only connection
php artisan firebase:test --test-connection

# Validate configuration
php artisan firebase:test --validate

# Show setup instructions
php artisan firebase:test --setup
```

### Manual Testing

1. **Check Environment Variables**
   ```bash
   php artisan tinker
   >>> env('FIREBASE_PROJECT_ID')
   ```

2. **Test Firebase Service**
   ```bash
   php artisan tinker
   >>> $firebaseService = app(App\Services\FirebaseService::class);
   >>> $firebaseService->validateConfiguration();
   ```

## 📱 Mobile App Integration

### Flutter Firebase Setup

#### 1. Add Dependencies

In your `pubspec.yaml`:

```yaml
dependencies:
  firebase_core: ^2.15.0
  firebase_messaging: ^14.6.2
  firebase_analytics: ^10.8.1
```

#### 2. Initialize Firebase

In your `main.dart`:

```dart
import 'package:firebase_core/firebase_core.dart';
import 'package:firebase_messaging/firebase_messaging.dart';

void main() async {
  WidgetsFlutterBinding.ensureInitialized();
  await Firebase.initializeApp();
  
  // Request notification permissions
  FirebaseMessaging messaging = FirebaseMessaging.instance;
  await messaging.requestPermission();
  
  runApp(MyApp());
}
```

#### 3. Handle Notifications

```dart
FirebaseMessaging.onMessage.listen((RemoteMessage message) {
  // Handle foreground notifications
  print('Got a message whilst app was in foreground!');
});

FirebaseMessaging.onMessageOpenedApp.listen((RemoteMessage message) {
  // Handle notifications when app is opened from terminated state
  print('Notification opened app');
});
```

### Deep Link Integration

#### Android Configuration

In `android/app/src/main/AndroidManifest.xml`:

```xml
<intent-filter>
    <action android:name="android.intent.action.VIEW" />
    <category android:name="android.intent.category.DEFAULT" />
    <category android:name="android.intent.category.BROWSABLE" />
    <data android:scheme="clxpro" android:host="app" />
</intent-filter>
```

#### iOS Configuration

In `ios/Runner/Info.plist`:

```xml
<key>CFBundleURLTypes</key>
<array>
    <dict>
        <key>CFBundleTypeRole</key>
        <string>Editor</string>
        <key>CFBundleURLName</key>
        <string>com.clxpro.app</string>
        <key>CFBundleURLSchemes</key>
        <array>
            <string>clxpro</string>
        </array>
    </dict>
</array>
```

## 🔐 Security Considerations

### Service Account Permissions

1. **Limit Service Account Scope**: Only grant necessary permissions
2. **Use Firebase Security Rules**: Configure database and storage rules
3. **Environment Variable Security**: Never commit `.env` files to version control

### Firebase Security Rules

#### Realtime Database Rules

```json
{
  "rules": {
    ".read": false,
    ".write": false,
    "notifications": {
      ".read": false,
      ".write": "auth != null && auth.token.admin == true"
    }
  }
}
```

#### Storage Rules

```javascript
rules_version = '2';
service firebase.storage {
  match /b/{bucket}/o {
    match /{allPaths=**} {
      allow read, write: if false;
    }
  }
}
```

## 🚨 Troubleshooting

### Common Issues

#### 1. "Invalid Credentials" Error

**Cause**: Incorrect service account key or missing permissions
**Solution**: 
- Regenerate service account key
- Ensure service account has Firebase Admin SDK permissions

#### 2. "Server Key Missing" Error

**Cause**: FIREBASE_SERVER_KEY not set in environment
**Solution**: Add server key from Firebase Console > Project Settings > Cloud Messaging

#### 3. "Project Not Found" Error

**Cause**: Incorrect project ID or Firebase not initialized
**Solution**: 
- Verify project ID in `.env`
- Ensure Firebase project exists and is active

#### 4. Mobile App Not Receiving Notifications

**Cause**: FCM token not registered or configuration issues
**Solution**:
- Check FCM token registration in mobile app
- Verify Firebase configuration in mobile app
- Check network connectivity and permissions

### Debug Commands

```bash
# Check environment variables
php artisan env

# Test Firebase connection
php artisan firebase:test --test-connection

# View Firebase configuration status
php artisan tinker
>>> app(App\Services\FirebaseConfigurationService::class)->getConfigurationStatus()
```

## 📊 Monitoring and Maintenance

### Firebase Console Monitoring

1. **FCM Analytics**: Monitor message delivery and engagement
2. **Error Reporting**: Track and resolve Firebase-related errors
3. **Usage Quotas**: Monitor API usage and quotas

### Application Monitoring

1. **Log Firebase Operations**: Log all Firebase operations for debugging
2. **Monitor Notification Delivery**: Track delivery rates and failures
3. **Performance Monitoring**: Monitor Firebase SDK performance

### Regular Maintenance

1. **Rotate Service Account Keys**: Regularly regenerate service account keys
2. **Update Dependencies**: Keep Firebase SDKs updated
3. **Review Security Rules**: Regularly audit and update security rules
4. **Monitor Quotas**: Watch for approaching usage limits

## 📞 Support

For additional help:

1. **Firebase Documentation**: [firebase.google.com/docs](https://firebase.google.com/docs)
2. **Laravel Firebase SDK**: [github.com/kreait/laravel-firebase](https://github.com/kreait/laravel-firebase)
3. **Flutter Firebase**: [firebase.flutter.dev](https://firebase.flutter.dev)

## 🎯 Next Steps

After completing Firebase setup:

1. **Configure Mobile App**: Set up Firebase in your Flutter application
2. **Test Notifications**: Send test notifications to verify setup
3. **Implement Deep Links**: Configure deep link handling in mobile app
4. **Set Up Queue Workers**: Configure background processing for notifications
5. **Deploy to Production**: Prepare for production deployment

---

**Note**: Keep your Firebase credentials secure and never share them publicly. Always use environment variables to store sensitive information.
