# Admin Profile Management

## Overview

The Admin Profile Management feature allows administrators to manage their personal account information and security settings within the CLXpro admin dashboard.

## Features

### 1. Profile Viewing
- View complete admin profile information
- Display account status and member since date
- Show admin privileges and permissions

### 2. Profile Editing
- Update full name
- Change email address (with validation)
- Add or update mobile number
- Real-time form validation

### 3. Password Management
- Change admin password securely
- Current password verification required
- Password confirmation validation
- Secure password hashing

### 4. User Interface
- Clean, responsive design matching admin theme
- Dropdown menu for quick access to profile actions
- Success/error message feedback
- Mobile-friendly responsive layout

## Access Methods

### Method 1: User Dropdown Menu
1. Click on the admin avatar in the top-right header
2. Select "My Profile" to view profile
3. Select "Edit Profile" to edit profile

### Method 2: Direct URLs
- **View Profile**: `/admin/profile`
- **Edit Profile**: `/admin/profile/edit`
- **Update Password**: `/admin/profile/password` (POST)

## Routes

```php
// Profile Routes
Route::get('/profile', [ProfileController::class, 'show'])->name('profile.show');
Route::get('/profile/edit', [ProfileController::class, 'edit'])->name('profile.edit');
Route::put('/profile', [ProfileController::class, 'update'])->name('profile.update');
Route::put('/profile/password', [ProfileController::class, 'updatePassword'])->name('profile.password.update');
```

## Controllers

### ProfileController

**Location**: `app/Http/Controllers/Admin/ProfileController.php`

**Methods**:
- `show()` - Display admin profile
- `edit()` - Show edit form
- `update()` - Update profile information
- `updatePassword()` - Update admin password

## Views

### Profile Show View
**Location**: `resources/views/admin/profile/show.blade.php`

**Features**:
- Profile information display
- Account details section
- Quick actions panel
- Password change form
- Admin privileges overview

### Profile Edit View
**Location**: `resources/views/admin/profile/edit.blade.php`

**Features**:
- Edit form with validation
- Profile information preview
- Form validation feedback
- Cancel and save buttons

## Security Features

### 1. Authentication Required
- All profile routes require admin authentication
- Protected by `CheckAdmin` middleware

### 2. Email Uniqueness
- Email validation ensures no duplicate emails
- Excludes current user from uniqueness check

### 3. Password Security
- Current password verification required
- Minimum 8 character password requirement
- Password confirmation validation
- Secure password hashing with bcrypt

### 4. CSRF Protection
- All forms include CSRF tokens
- Protected against cross-site request forgery

## Validation Rules

### Profile Update
```php
'name' => ['required', 'string', 'max:255'],
'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,' . $user->id],
'mobile_number' => ['nullable', 'string', 'max:20'],
```

### Password Update
```php
'current_password' => ['required', 'string'],
'password' => ['required', 'string', 'min:8', 'confirmed'],
```

## Database Fields Used

The profile management uses the following user model fields:
- `name` - Admin's full name
- `email` - Admin's email address
- `mobile_number` - Admin's mobile number (optional)
- `password` - Admin's password (hashed)

## Integration with Admin Layout

### User Dropdown Menu
The admin layout includes a user dropdown menu that provides:
- Quick access to profile viewing
- Direct link to profile editing
- Logout functionality

### Theme Consistency
- Matches existing admin dashboard theme
- Supports both light and dark themes
- Responsive design for mobile devices

## Usage Examples

### Viewing Profile
```php
// In browser: /admin/profile
// Shows complete admin profile with account information
```

### Editing Profile
```php
// In browser: /admin/profile/edit
// Allows updating name, email, and mobile number
```

### Changing Password
```php
// Submit form to: /admin/profile/password
// Requires current password + new password + confirmation
```

## Error Handling

### Validation Errors
- Displayed inline with form fields
- Clear error messages for user guidance
- Form retains user input on validation failure

### Password Errors
- Current password incorrect
- Password too short
- Password confirmation mismatch

## Success Feedback

### Success Messages
- Profile updated successfully
- Password changed successfully
- Displayed as toast notifications
- Auto-dismiss after 5 seconds

## Mobile Responsiveness

### Responsive Design
- Grid layout adapts to screen size
- Dropdown menu works on touch devices
- Form fields resize appropriately
- Touch-friendly button sizes

## Browser Compatibility

### Supported Browsers
- Chrome (Latest)
- Firefox (Latest)
- Safari (Latest)
- Edge (Latest)

### JavaScript Requirements
- Modern browser with ES6+ support
- LocalStorage for theme preferences
- DOM manipulation for dropdown functionality

## Future Enhancements

### Potential Features
- Profile picture upload
- Two-factor authentication
- Login history tracking
- Admin activity logs
- Password strength indicator
- Email verification system

## Troubleshooting

### Common Issues

**Profile not saving changes**:
- Check internet connection
- Verify form validation
- Ensure CSRF token is present

**Password change failing**:
- Verify current password is correct
- Check password meets requirements
- Ensure password confirmation matches

**Dropdown not working**:
- Check JavaScript is enabled
- Verify no JavaScript errors in console
- Ensure DOM elements are properly loaded

## Support

For issues related to admin profile management:
1. Check browser console for JavaScript errors
2. Verify form validation messages
3. Ensure proper authentication
4. Check server logs for backend errors
